/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef EXTERNALSTORAGEMODEL_H
#define EXTERNALSTORAGEMODEL_H

#include <QAbstractListModel>
#include <partitionmanager.h>
#include <functional>

class SYSTEMSETTINGS_EXPORT ExternalStorageModel : public QAbstractListModel
{
    Q_OBJECT
    Q_PROPERTY(bool noStorages READ noStorages NOTIFY storageCountChanged)
    Q_PROPERTY(bool noSdCard READ noSdCard NOTIFY storageCountChanged)
    Q_PROPERTY(bool storagesReady READ storagesReady NOTIFY storagesReadyChanged)

public:
    enum {
        StorageLabelRole,
        StorageConnectionBusRole,
        StoragePartitionRole,
        BytesAvailableRole,
        BytesTotalRole
    };

    explicit ExternalStorageModel(QObject *parent = nullptr);
    ~ExternalStorageModel();

    bool noStorages() const;
    bool noSdCard() const;
    bool storagesReady() const;

    Q_INVOKABLE void refresh();

    QHash<int, QByteArray> roleNames() const override;
    int rowCount(const QModelIndex &parent = QModelIndex()) const override;
    QVariant data(const QModelIndex &index, int role) const override;

signals:
    void storageCountChanged();
    void storagesReadyChanged();

private:
    void setStoragesReady(bool value);

    QVector<Partition>::const_iterator findExternalStorage(const Partition &storagePartition);
    void onStorageAdded(const Partition &partition);
    void onStorageRemoved(const Partition &partition);

    using ChildPartitionHandler = std::function<void(QVector<Partition> &, const QVector<Partition>::iterator &)>;
    void handleChildPartition(const Partition &partition, ChildPartitionHandler&& handler);
    void onPartitionAdded(const Partition &partition);
    void onPartitionRemoved(const Partition &partition);
    void onPartitionChanged(const Partition &partition);

    QExplicitlySharedDataPointer<PartitionManagerPrivate> m_manager;
    bool m_storagesReady;
    QVector<Partition> m_externalStorages;
    QHash<QString, QVector<Partition>> m_childrenByStorageDevicePath;
};

#endif // EXTERNALSTORAGEMODEL_H
