/*
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#ifndef __STREAMCAMERA__
#define __STREAMCAMERA__

/**
 * \file streamcamera.h
 * Video camera support and data types.
 */

#include <sys/types.h>

#include <string>
#include <vector>
#include <memory>
#include <mutex>

#include "streamcamera-types.h"
#include "streamcamera-fraction.h"

namespace Aurora {
namespace StreamCamera {

static constexpr std::string_view InternalFrontCameraName("Internal front camera");
static constexpr std::string_view InternalRearCameraName("Internal rear camera");
static constexpr std::string_view InternalCameraName("Internal camera");
static constexpr std::string_view ExternalCameraName("External camera");

/// Camera facing
enum class CameraFacing
{
    Unknown,                    ///< For external cameras
    Front,                      ///< The camera is a front-facing camera
    Rear,                       ///< The camera is a rear-facing camera
    Screen                      ///< Virtual camera, allowing access to screen content.
                                ///  The application must have ScreenCapture permission,
                                ///  otherwise such entries will not be listed.
};

/// Video stream settings
// \see CameraManager::queryCapabilities()
struct CameraCapability
{
    unsigned int width;         ///< Frame width
    unsigned int height;        ///< Frame height
    unsigned int fps;           ///< Frame rate
};

template<typename T>
struct Range
{
    T min{}, max{}, step{};

    bool isNull() const { return min == T{} && max == T{}; }
    bool isEmpty() const { return min == max; }
};

struct CameraCapabilityRanges
{
    Range<unsigned int> width{};         ///< Frame width range
    Range<unsigned int> height{};        ///< Frame height range
    Range<Fraction<unsigned int>> fps; ///< Frame rate range

    bool isValid() const
    {
        return !width.isNull() || !height.isNull() || !fps.isNull();
    }
};

struct CameraCapabilityEx
{
    unsigned int width;         ///< Frame width
    unsigned int height;        ///< Frame height
    Fraction<unsigned int> fps; ///< Frame rate

    bool isValid() const
    {
        return width > 0
            && height > 0
            && fps.isValid()
            && fps > decltype(fps) { 0, 1 };
    }
};

/// Camera capability priority hint
/// \see Camera::findClosestCapability() and CameraManager::findClosestCapability()
enum class CameraCapabilityPriority
{
    WidthHeightFps,
    HeightWidthFps,
    FpsWidthHeight,
    FpsHeightWidth
};

/// Information about camera
// \see CameraManager::getCameraInfo()
struct CameraInfo
{
    std::string id;             ///< Unique id
    std::string name;           ///< Human-readable camera name
    std::string provider;       ///< Camera plugin name
    CameraFacing facing;        ///< Camera facing
    unsigned int mountAngle;    ///< An angle at which the camera is rotated in
    std::string metadata;       ///< Camera metadata, like angle of view or aperture
};

/// A template to make FourCC value
template<char a, char b, char c, char d>
constexpr uint32_t FourCC{static_cast<uint32_t>(a)
        | (static_cast<uint32_t>(b) << 8)
        | (static_cast<uint32_t>(c) << 16)
        | (static_cast<uint32_t>(d) << 24)};

/// Frame pixel format
// \see CameraManager::getSupportedPixelFormats()
enum class PixelFormat
{
    Invalid = 0,                ///< Invalid frame

    // YCbCr 4:2:0
    YUV420Planar,               ///< \deprecated The same as I420.
    YUV420SemiPlanar,           ///< \deprecated The same as NV12.
    YCbCrFlexible = 0xff,       ///< \deprecated The actual format is described by YCbCrFrame.

    // YUV-like:
    NV12 = FourCC<'N', 'V', '1', '2'>,
    NV21 = FourCC<'N', 'V', '2', '1'>,
    I420 = FourCC<'Y', 'U', '1', '2'>,
    YUYV = FourCC<'Y', 'U', 'Y', 'V'>,
    UYVY = FourCC<'U', 'Y', 'V', 'Y'>,

    // RGB-like:
    XRGB32 = FourCC<'B', 'X', '2', '4'>,  ///< B:G:R:X in memory or 0xXXRRGGBB little-endian as a 32-bit value

    // Encoded:
    MJPEG = FourCC<'M', 'J', 'P', 'G'>,
    JPEG = FourCC<'J', 'P', 'E', 'G'>,
    H264 = FourCC<'H', '2', '6', '4'>,
    HEVC = FourCC<'H', 'E', 'V', 'C'>,
    VP8 = FourCC<'V', 'P', '8', '0'>,
    VP9 = FourCC<'V', 'P', '9', '0'>,

    // Special values:
    Mappable = 0xfffb,
    NotMappable = 0xfffc,
    Raw = 0xfffd,
    Encoded = 0xfffe,
    Preferred = 0xffff
};

std::string toString(PixelFormat pixelFormat);

struct PixelFormatDescription
{
    PixelFormat pixelFormat;

    enum Flag : uint64_t
    {
        ZeroCopy            = 1,
        RequiresConverting  = 1 << 1,
    };
    uint64_t flags;

    bool operator==(const PixelFormatDescription &o) const
    {
        return pixelFormat == o.pixelFormat && flags == o.flags;
    }
};

/// The type of the graphic buffer
// \see GraphicBuffer::handle
enum class HandleType
{
    NoHandle,                   ///< The buffer is a plain image
    ANativeWindowBuffer,        ///< The handle is described by the ANDROID_image_native_buffer extension
    EGL = ANativeWindowBuffer,  ///< An alias for backward comatibility
    GBMImportData,              ///< The handle is the pointer to `struct gbm_import_fd_modifier_data`, see libgbm docs.
};

/**
 * The descriptor of YUV-like 4:2:0 video frame.
 *
 * This is a flexible format, the exact format can be guessed from \a cStride and
 * \a chromaStep values. For I420 \a chromaStep == 1. If \a chromaStep is 2,
 * the format is likely NV12. Examples:
 *
 * Planar I420:
 * \code
 *     y y y y <- luma plane
 *     cb  cb  <- cb plane
 *     cr  cr  <- cr plane
 * \endcode
 *
 * Semi-planar NV12:
 * \code
 *     y y y y <- luma plane
 *     cb  cr  <- cb plane, cr = cb + 1
 *     cb  cr
 * \endcode
 *
 * \see GraphicBuffer::mapYCbCr()
 */
struct YCbCrFrame
{
    const uint8_t *y;           ///< Pointer to luma values
    const uint8_t *cb;          ///< Pointer to cromaB values
    const uint8_t *cr;          ///< Pointer to cromaR values
    uint16_t yStride;           ///< Luma stride
    uint16_t cStride;           ///< Chroma stride
    uint16_t chromaStep;        ///< A step between chroma values.
                                ///< chromaStep == 1 for planar format
                                ///< chromaStep == 2 for semi-planar formats.
    uint16_t width;             ///< Frame width
    uint16_t height;            ///< Frame height
    uint64_t timestampUs;       ///< Monotonically increasing timestamp in microseconds
};

/// The descriptor of video frame
// \see GraphicBuffer::map()
struct RawImageFrame
{
    const uint8_t *data;        ///< Pointer to frame data
    size_t size;                ///< Size of frame data
    PixelFormat pixelFormat;    ///< Actual frame pixel format
    uint16_t width;             ///< Frame width
    uint16_t height;            ///< Frame height
    uint64_t timestampUs;       ///< Monotonically increasing timestamp in microseconds
};

/// The descriptor of video frame image data
struct Image
{
    PixelFormat pixelFormat;    ///< Actual pixel format, defines planes and order of components in them

    struct Plane
    {
        uint16_t width;
        uint16_t height;

        union {
            uint16_t stride;    ///< Row stride of a raw image plane
            uint32_t size;      ///< Size of encoded data buffer
        };

        const uint8_t *data;    ///< Raw image plane or encoded data
    };

    Plane planes[4];  ///< Image planes
};

/// The descriptor of video frame
// \see GraphicBuffer::mapFrame()
struct Frame
{
    Image image;
    uint64_t timestampUs;       ///< Monotonically increasing timestamp in microseconds
};

/**
 * \brief Graphic buffer descriptor.
 *
 * Graphic buffers are produced by the Camera and capable to be rendered via
 * EGL or mapped in memory to access to video data.
 */
class GraphicBuffer
{
public:
    virtual ~GraphicBuffer() = default;

    uint16_t width;                     ///< Picture width
    uint16_t height;                    ///< Picture height

    uint64_t timestampUs = -1;          ///< Capture timestamp in microseconds

    PixelFormat pixelFormat = PixelFormat::Invalid;     ///< Pixel format for video data

    const void *handle = nullptr;       ///< Hardware-specific handle for the underlying media buffer
    HandleType handleType = HandleType::NoHandle; ///< Handle type

    /**
     * \brief  Map YUV video frame.
     *
     * The frame should have PixelFormat <= PixelFormat::YCbCrFlexible.
     *
     * \returns A pointer to data descriptor or nullptr if data cannot be mapped in this way.
     */
    virtual std::shared_ptr<const YCbCrFrame> mapYCbCr() = 0;

    /**
     * \brief Map video frame.
     *
     * \returns A pointer to data descriptor or nullptr if data cannot be mapped in this way.
     */
    virtual std::shared_ptr<const RawImageFrame> map() = 0;

    /**
     * \brief Map video frame to the Image struct.
     *
     * \returns A pointer to data descriptor or nullptr if data cannot be mapped in this way.
     */
    virtual std::shared_ptr<const Frame> mapFrame() = 0;

    /**
     * \brief Buffer rotation.
     *
     * \returns An angle the buffer should be rotated to properly show the
     * contents. Possible values are 0, 90, 180, 270. Considers device
     * orientation from sensors, CameraInfo.mountAngle and CameraInfo.facing.
     * Apply this value on other side of a video conference.
     */
    virtual uint16_t rotation() const = 0;
};

/// Camera parameter
enum class CameraParameter : unsigned int
{
    FlashMode,                    ///< Flashlight mode ["off", "auto", "on", "red-eye", "torch"]
                                  ///< There is no flash on this camera if getParameterRange()
                                  ///< for this value returns an empty string.
    Last,                         ///< A helper to iterate parameters
    Invalid = Last,               ///< Deprecated, do not use.
};

/**
 * \brief Callbacks for the camera.
 *
 * The user must implement this class and provide callbacks for the camera
 * with Camera::setListener.
 */
class CameraListener
{
public:
    virtual ~CameraListener() = default;

    /**
     * \brief Called when a graphics buffer is captured.
     *
     * Called on a different thread. The time spent in this function should be mimimized.
     *
     * \param[out] buffer A captured GraphicBuffer.
     */
    virtual void onCameraFrame(std::shared_ptr<GraphicBuffer> buffer) = 0;

    /// Called when a camera error occurs.
    virtual void onCameraError(const std::string &errorDescription) = 0;

    /// Called when a hardware-initiated change of a parameter occurs. For future purposes.
    virtual void onCameraParameterChanged(CameraParameter, const std::string &value) = 0;
};

/**
 * \brief Camera
 */
class Camera
{
public:
    virtual ~Camera() = default;
    /**
     * \brief Get camera info.
     *
     * \param[out] Reference to the CameraInfo structure for the output.
     * \returns True if the camera information was successfully read.
     */
    virtual bool getInfo(CameraInfo &info) = 0;

    /**
     * \brief List supported pixel formats.
     *
     * \returns A list of supported pixel formats, or an empty list if an error occurs.
     */
    virtual std::vector<PixelFormat> getSupportedPixelFormats() = 0;

    /**
     * \brief Start capture.
     *
     * \param[in] cap Output video parameters.
     * \param[in] format The desired pixel format for graphic buffers.
     * \returns True if the camera was started successfully, false otherwise.
     */
    virtual bool startCapture(const CameraCapability &cap,
                              PixelFormat format = PixelFormat::YCbCrFlexible) = 0;

    /**
     * \brief Stop capture.
     * \returns True if the camera already stopped or stopped successfully, otherwise false.
     */
    virtual bool stopCapture() = 0;

    /**
     * \brief Check if the camera is running.
     * \returns True if the camera is running, false otherwise.
     */
    virtual bool captureStarted() const = 0;

    /**
     * \brief Get possible values for a parameter.
     *
     * \param[in] param Parameter index
     * \returns Return value syntax for the getParameterRange():<br>
     *   * "v1,v2,v3" for a possible set of values, for example "1,2" or "false,true"<br>
     *     A pre-C++20 example of parsing a discrete range of hypothetical DiscreteInt:
     *
     *         #include <string>
     *         #include <vector>
     *         #include <iostream>
     *
     *         std::vector<std::string> split(std::string in)
     *         {
     *             std::vector<std::string> tokens;
     *
     *             if (in.length() == 0)
     *                 return {};
     *
     *             std::size_t start = 0, end = 0;
     *             for (auto v : in) {
     *                 if (v == ',') {
     *                     tokens.push_back(in.substr(start, end - start));
     *                     start = end + 1;
     *                 }
     *                 end++;
     *             }
     *             if (start != end)
     *                 tokens.push_back(in.substr(start, end - start));
     *
     *             return tokens;
     *         }
     *
     *         std::vector<int> convert(std::vector<std::string> in)
     *         {
     *             std::vector<int> result;
     *             for (auto v : in)
     *                 result.push_back(std::stoi(v));
     *
     *             return result;
     *         }
     *
     *         ...
     *
     *             std::string range = camera.getParameterRange(CameraParameter::DiscreteInt);
     *             std::vector<int> rangeInt = convert(split(range));
     *
     *   * "v1-v10" for a сontinuous value, for example "1-10" or "0.0-1.0" for float values.<br>
     *     An example of parsing such a range for a hypothetical FloatValue:
     *
     *         float minValue, maxValue;
     *         std::string range = camera->getParameterRange(CameraParameter::FloatValue);
     *         bool success = sscanf(range.c_str(), "%f-%f, &minV, &maxV) == 2;
     *
     *   * "-"  for a read-only value
     *   * "x" for a rectangle (the value is "(x,y,w,h)")
     *   * empty string if an error occurs
     */
    virtual std::string getParameterRange(CameraParameter param) = 0;

    /**
     * \brief Get current value for a parameter.
     *
     * \param[in] param Parameter index
     * \returns Return value syntax for the getParameter():<br>
     *   * "v"
     *   * "(x,y,w,h)" for a rectangle
     *   * empty string if an error occurs
     */
    virtual std::string getParameter(CameraParameter param) = 0;

    /**
     * \brief Set value of a parameter.
     *
     * \param[in] param Parameter index
     * \param[in] value
     *   * "v" for a discrete or continuous value
     *   * "(x,y,w,h)" for a rectangle
     * \returns True is the value is accepted.
     */
    virtual bool setParameter(CameraParameter param, const std::string &value) = 0;

    /**
     * \brief Set callbacks.
     * 
     * Guarantees only that calling callbacks and changing listeners are mutually exclusive.
     * 
     * \param[in] listener An object with callbacks. Use nullptr to unset before destroying a listener.
     */
    virtual void setListener(CameraListener *listener)
    {
        std::lock_guard guard(m_cameraListenerMutex);
        m_cameraListener = listener;
    }

    /**
     * \brief List supported pixel formats with the additional information.
     *
     * \returns A list of supported pixel formats, or an empty list if an error occurs.
     */
    virtual std::vector<PixelFormatDescription> getSupportedFormats() = 0;

    /**
     * \brief Check if a format is supported considering the additional information.
     */
    virtual bool isFormatSupported(const PixelFormatDescription &formatDesc) = 0;

    /**
     * \brief Check if a format is supported.
     */
    virtual bool isFormatSupported(PixelFormat pixelFormat) = 0;

    /** Query camera capability ranges for a specified format.
     * 
     * \param[in] format Pixel format capability ranges are queried.
     * \param[out] capRanges Reference to a \a CameraCapabilityRanges structure to fill in.
     * \returns true on success and false otherwise, including the case when the feature is unsupported.
     */
    virtual bool queryCapabilityRanges(PixelFormat format,
                                       CameraCapabilityRanges &capRanges) = 0;

    /**
     * \brief Find the closest camera capability for a specified format.
     *
     * \param[in] format Pixel format capability ranges are queried.
     * \param[in] desired Capability reference to find the closest one.
     * \param[out] found Capability reference to find the closest one.
     * \param[in] priorityHint Order of capability fields from the most important to less ones.
     * \returns true on success and false otherwise, \a found isn't modified in the latter case.
     */
    virtual bool findClosestCapability(PixelFormat format,
                                       const CameraCapabilityEx &desired,
                                       CameraCapabilityEx &found,
                                       CameraCapabilityPriority priorityHint = CameraCapabilityPriority::WidthHeightFps) = 0;

    /**
     * \brief Start capture.
     *
     * \param[in] cap Output video parameters.
     * \param[in] format The desired pixel format for graphic buffers.
     * \returns True if the camera was started successfully, false otherwise.
     */
    virtual bool startCapture(const CameraCapabilityEx &cap,
                              PixelFormat format = PixelFormat::YCbCrFlexible) = 0;

    /**
     * \brief Query camera capabilities.
     *
     * \param[out] Reference to a list of CameraCapability structures to fill in.
     * \returns True if the capabilites were successfully read.
     */
    virtual bool queryCapabilities(std::vector<CameraCapability> &caps) = 0;

protected:
    CameraListener *m_cameraListener = nullptr;
    std::recursive_mutex m_cameraListenerMutex;
};

/**
 * \brief Callbacks for the camera manager.
 *
 * The user must implement this class and provide callbacks for the camera manager
 * with CameraManager::setListener.
 */
class CameraManagerListener
{
public:
    virtual ~CameraManagerListener() = default;

    /**
     * \brief Called when a new camera appears in the system.
     */
    virtual void onCameraAdded(const std::string &cameraId) = 0;

    /**
     * \brief Called when a camera disappears from the system.
     */
    virtual void onCameraRemoved(const std::string &cameraId) = 0;
};

/**
 * Camera manager
 */
class CameraManager
{
public:
    virtual ~CameraManager() = default;
    /**
     * \brief (Re-)initilize the camera manager.
     *
     * This method can be used to search for new connected devices. It is not necessary
     * to call this method before first accessing other CameraManager methods.
     */
    virtual bool init() = 0;

    /**
     * \brief Get the number of cameras found.
     */
    virtual int getNumberOfCameras() = 0;

    /**
     * \brief Get camera info.
     *
     * \param[in] num Camera number starting from zero.
     * \param[out] info Reference to the CameraInfo structure for the output.
     * \returns True if the camera information was successfully read.
     */
    virtual bool getCameraInfo(unsigned int num, CameraInfo &info) = 0;

    /**
     * \brief Query camera capabilities.
     *
     * \param[in] cameraId The camera ID.
     * \param[out] Reference to a list of CameraCapability structures to fill in.
     * \returns True if the capabilites were successfully read.
     */
    virtual bool queryCapabilities(const std::string &cameraId,
                                   std::vector<CameraCapability> &caps) = 0;

    /**
     * \brief Open the camera.
     *
     * \param[in] cameraId Camera ID
     * \returns A pointer to the camera or nullptr in case of any error.
     */
    virtual std::shared_ptr<Camera> openCamera(const std::string &cameraId) = 0;

    /**
     * \param[in] cameraId Camera ID
     * \brief List supported pixel formats with the additional information.
     *
     * \returns A list of supported pixel formats, or an empty list if an error occurs.
     */
    virtual std::vector<PixelFormatDescription> getSupportedFormats(const std::string &cameraId) = 0;

    /** Query camera capability ranges.
     * 
     * \param[in] cameraId Camera ID
     * \param[in] format Pixel format capability ranges are queried.
     * \param[out] capRanges Reference to a \a CameraCapabilityRanges structure to fill in.
     */
    virtual bool queryCapabilityRanges(const std::string &cameraId, PixelFormat format,
                                       CameraCapabilityRanges &capRanges) = 0;

    /**
     * \brief Find the closest camera capability for a specified format.
     *
     * \param[in] cameraId Camera ID
     * \param[in] format Pixel format capability ranges are queried.
     * \param[in] desired Capability reference to find the closest one.
     * \param[out] found Capability reference to find the closest one.
     * \param[in] priorityHint Order of capability fields from the most important to less ones.
     * \returns true on success and false otherwise, \a found isn't modified in the latter case.
     */
    virtual bool findClosestCapability(const std::string &cameraId,
                                       PixelFormat format,
                                       const CameraCapabilityEx &desired,
                                       CameraCapabilityEx &found,
                                       CameraCapabilityPriority priorityHint = CameraCapabilityPriority::WidthHeightFps) = 0;

    /**
     * \brief Set callbacks.
     * 
     * Guarantees only that calling callbacks and changing listeners are mutually exclusive.
     * 
     * \param[in] listener An object with callbacks. Use nullptr to unset before destroying a listener.
     */
    virtual void setListener(CameraManagerListener *listener)
    {
        std::lock_guard guard(m_cameraManagerListenerMutex);
        m_cameraManagerListener = listener;
    }

protected:
    CameraManagerListener *m_cameraManagerListener = nullptr;
    std::recursive_mutex m_cameraManagerListenerMutex;
};

} // namespace StreamCamera
} // namespace Aurora

/**
 * \brief Constructor for the root CameraManager.
 *
 * \returns A pointer to statically allocated and initialized CameraManager.
 */
extern "C" __attribute__((visibility("default")))
Aurora::StreamCamera::CameraManager *StreamCameraManager(void);

#endif /* __STREAMCAMERA__ */
/* vim: set ts=4 et sw=4 tw=80: */
