/*
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#pragma once

#include <cstdio>
#include <sstream>

/**
 * \file streamcamera-types.h
 * Data types.
 */

namespace Aurora {
namespace StreamCamera {

/**
 * Size value.
 * \since v1.5.0
 */
struct Size
{
    int width;
    int height;

    Size() : Size(-1, -1)
    {
    }

    Size(int width, int height)
    {
        this->width = width;
        this->height = height;
    }

    bool isValid() const
    {
        return width >= 0 && height >= 0;
    }

    std::string toString() const
    {
        std::ostringstream s;
        s << "(" << width << "," << height << ")";
        return s.str();
    }

    bool fromString(const std::string &s)
    {
        return std::sscanf(s.c_str(), "(%d,%d)", &width, &height) == 2;
    }
};

/**
 * Size range value.
 * \since v1.5.0
 */
struct SizeRange
{
    Size min;
    Size max;

    SizeRange()
    {
    }

    SizeRange(const Size &min, const Size &max)
    {
        this->min = min;
        this->max = max;
    }

    bool isValid() const
    {
        return min.isValid() && max.isValid();
    }

    std::string toString() const
    {
        std::ostringstream s;
        s << min.toString() << "-" << max.toString();
        return s.str();
    }

    bool fromString(const std::string &s)
    {
        return std::sscanf(s.c_str(), "(%d,%d)-(%d,%d)",
                &min.width, &min.height, &max.width, &max.height) == 4;
    }
};

/**
 * Rectangle value.
 * \since v1.5.0
 */
struct Rectangle
{
    int top;
    int left;
    int width;
    int height;

    Rectangle() : Rectangle(-1, -1, -1, -1)
    {
    }

    Rectangle(int top, int left, int width, int height)
    {
        this->top = top;
        this->left = left;
        this->width = width;
        this->height = height;
    }

    bool isValid() const
    {
        return top >= 0 && left >= 0 &&  width >= 0 && height >= 0;
    }

    std::string toString() const
    {
        std::ostringstream s;
        s << "(" << top << "," << left << "," << width << "," << height << ")";
        return s.str();
    }

    bool fromString(const std::string &s)
    {
        return std::sscanf(s.c_str(), "(%d,%d,%d,%d)",
                &top, &left, &width, &height) == 4;
    }
};

} // namespace StreamCamera
} // namespace Aurora
