/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#pragma once

#include "streamcamera-codec.h"
#include "streamcamera.h"

namespace Aurora {
namespace StreamCamera {

/// Encoded video metadata
class VideoStreamMetadata
{
public:
    int width() const;
    int height() const;
    int level() const;
    int profile() const;
    const std::vector<uint8_t> sps() const;
    const std::vector<uint8_t> pps() const;

    // Copy & move are allowed
    VideoStreamMetadata(const VideoStreamMetadata &other);
    VideoStreamMetadata &operator=(const VideoStreamMetadata &other);
    VideoStreamMetadata(VideoStreamMetadata &&other);
    VideoStreamMetadata &operator=(VideoStreamMetadata &&other);

    ~VideoStreamMetadata();

private:
    VideoStreamMetadata();
    friend class VideoStreamParserImpl;
    class Impl;
    Impl *impl;
};

// Implementation forward declaration
class VideoStreamParserImpl;
class VideoStreamParserListener;

/**
 * \brief Video stream parser
 */
class VideoStreamParser
{
public:
    VideoStreamParser();
    ~VideoStreamParser();

    /**
     * @brief RequestType for parsing video stream data.
     *
     *  Type indicates the expected structure of the input data.
     */
    enum class RequestType {
        Default,                ///< No specific structure; parse as raw data.
        CompleteNALU,           ///< Input contains complete NAL Units.
        CompleteAccessUnit      ///< Input contains complete Access Units (frames).
    };

    /**
     * @brief ConversionType for parsing video stream data.
     *
     *  Type indicates conversion between AnnexB/AVCC.
     */
    enum class ConversionType {
        None                    ///< No conversion
    };

    /**
     * @brief Describes a Network Abstraction Layer Unit (NALU) in a video stream buffer.
     *
     * This struct stores the position and size of a NALU within a larger data buffer.
     *
     */
    struct NaluDesc
    {
        off_t offset;   ///< Byte offset of the NALU start in the buffer.
        size_t size;    ///< Size of the NALU in bytes.
    };

    /**
     * @brief Stream parser settings.
     */
    class Settings {
    public:
        ~Settings();
    private:
        Settings();
        friend class ParserSettingsBuilder;

        struct Impl;
        std::unique_ptr<Impl> impl;
    };

    /**
     * \brief Parse a video chunk.
     *
     * \param[in] data A pointer to encoded video unit
     * \param[in] size Encoded video unit size
     * \param[in] timestampUs A video frame timestamp
     * \param[in] requestType Flags containing information about the input buffer (NALU/frame completeness etc.)
     */
    bool parse(const uint8_t *data, size_t size, uint64_t timestampUs, VideoStreamParser::RequestType requestType);

    /**
     * \brief Get current Metadata value.
     *
     * \param[in] Video stream parameters pointer
     * \returns True if parameters have been set
     */
    VideoStreamMetadata getMetadata();

    /**
     * \brief Set callbacks. A thread-safe method to set the listener.
     *
     * Guarantees only that calling callbacks and changing listeners are mutually exclusive.
     *
     * \param[in] listener An object with callbacks. Use nullptr to unset before destroying a listener.
     */
    void lockAndSetListener(VideoStreamParserListener *listener);

    /**
     * \brief Apply parser settings.
     *
     * Optinonal.
     *
     * \param[in] settings. Video stream parser settings
     */
    void applySettings(std::unique_ptr<Settings> settings);

protected:
    std::unique_ptr<VideoStreamParserImpl> m_parser;
};

/**
 * \brief Callbacks for the stream parser.
 *
 * The user must implement this class and provide callbacks for the parser
 * with VideoStreamParser::lockAndSetListener.
 * \warning Callbacks execute synchronously during `VideoStreamParser::parse()`.
 *          Do NOT call `parse()` or `lockAndSetListener()` from callbacks.
 */
class VideoStreamParserListener
{
public:
    virtual ~VideoStreamParserListener() = default;

    /**
     * \brief Called when a new frame/access/EndOfStream unit is ready.
     *
     * Called on a different thread. The time spent in this function should be minimized.
     *
     * \param[out] data Pointer to encoded data buffer (owned by parser)
     * \param[out] size Encoded data size
     * \param[out] nalu vector of nalu decriptors
     * \param[out] timestampUs Video frame timestamp
     * \param[out] frameType Keyframe or not
     */
    virtual void onOutputAvailable(
        uint8_t *data,
        size_t size,
        const std::vector<VideoStreamParser::VideoStreamParser::NaluDesc> &nalu,
        uint64_t timestampUs,
        FrameType frameType)
        = 0;

    /// Called when a stream parameters have been changed.
    virtual void onStreamChanged(VideoStreamMetadata meta) = 0;

    /// Called when an parser error occurs.
    virtual void onParserError(const std::string &errorDescription) = 0;

    /// Called when an end-of-stream occurs.
    virtual void onParserEOS() = 0;
};

/**
 * \brief Stream parser settings builder.
 */
class ParserSettingsBuilder {
public:
    ParserSettingsBuilder();
    ~ParserSettingsBuilder();

    std::unique_ptr<VideoStreamParser::Settings> build();

private:
    struct Impl;
    std::unique_ptr<Impl> impl;
};

} // namespace StreamCamera
} // namespace Aurora
