/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#pragma once

#include <iostream>
#include <numeric>
#include <cmath>

namespace Aurora {
namespace StreamCamera {

template<typename T>
struct Fraction
{
    T numerator{};
    T denominator{1};

    bool isValid() const
    {
        return denominator > 0;
    }

    static Fraction normalize(const Fraction &o)
    {
        auto t = o;
        if constexpr (std::is_signed_v<T>) {
            if (t.denominator < 0) {
                t.denominator = -t.denominator;
                t.numerator = -t.numerator;
            }
        }
        t.reduce();
        return t;
    }

    constexpr Fraction() {}
    constexpr Fraction(const T &n, const T &d = 1)
        : numerator(n)
        , denominator(d)
    {}

    void reduce()
    {
        if (numerator != 0) {
            const T d = std::gcd(denominator, numerator);
            numerator /= d;
            denominator /= d;
        } else {
            denominator = 1;
        }
    }

    Fraction &operator=(const T &n)
    {
        numerator = n;
        denominator = 1;
        return *this;
    }

    operator bool() const
    {
        return numerator != 0;
    }

    operator int() const
    {
        return std::round(double(numerator) / denominator);
    }

    operator unsigned int() const
    {
        const auto t = normalize(*this);
        return std::round(double(t.numerator) / t.denominator);
    }

    bool operator==(const Fraction &o) const
    {
        return (numerator * o.denominator) == (o.numerator * denominator);
    }

    bool operator!=(const Fraction &o) const
    {
        return !(*this == o);
    }

    bool operator<(const Fraction &o) const
    {
        const auto a = normalize(*this);
        const auto b = normalize(o);
        return (a.numerator * b.denominator) < (b.numerator * a.denominator);
    }

    bool operator>(const Fraction &o) const
    {
        const auto a = normalize(*this);
        const auto b = normalize(o);
        return (a.numerator * b.denominator) > (b.numerator * a.denominator);
    }

    bool operator<=(const Fraction &o) const
    {
        return !(*this > o);
    }

    bool operator>=(const Fraction &o) const
    {
        return !(*this < o);
    }

    const Fraction &operator-() const
    {
        numerator = -numerator;
        return *this;
    }

    const Fraction &operator+() const
    {
        numerator = +numerator;
        return *this;
    }

    const Fraction &operator+=(const Fraction &o)
    {
        numerator = numerator * o.denominator + o.numerator * denominator;
        denominator *= o.denominator;
        return *this;
    }

    const Fraction &operator-=(const Fraction &o)
    {
        numerator = numerator * o.denominator - o.numerator * denominator;
        denominator *= o.denominator;
        return *this;
    }

    const Fraction &operator*=(const Fraction &o)
    {
        numerator *= o.numerator;
        denominator *= o.denominator;
        return *this;
    }

    const Fraction &operator/=(const Fraction &o)
    {
        numerator *= o.denominator;
        denominator *= o.numerator;
        return *this;
    }

    const Fraction &operator%=(const Fraction &o)
    {
        *this /= o;
        numerator %= denominator;
        *this *= o;
        return *this;
    }
};

template<typename T, typename U>
bool operator==(const Fraction<T> &a, const U &b)
{
    return a == Fraction<T>(b);
}

template<typename T, typename U>
bool operator==(const U &a, const Fraction<T> &b)
{
    return Fraction<T>(a) == b;
}

template<typename T, typename U>
bool operator!=(const Fraction<T> &a, const U &b)
{
    return !(a == b);
}

template<typename T, typename U>
bool operator!=(const U &a, const Fraction<T> &b)
{
    return !(a == b);
}

template<typename T>
Fraction<T> operator+(const Fraction<T> &a, const Fraction<T> &b)
{
    auto c = a;
    c += b;
    return c;
}

template<typename T, typename U>
Fraction<T> operator+(const Fraction<T> &a, const U &b)
{
    return a + Fraction<T>(b);
}

template<typename T, typename U>
Fraction<T> operator+(const U &a, const Fraction<T> &b)
{
    return Fraction<T>(a) + b;
}

template<typename T>
Fraction<T> operator-(const Fraction<T> &a, const Fraction<T> &b)
{
    auto c = a;
    c -= b;
    return c;
}

template<typename T, typename U>
Fraction<T> operator-(const Fraction<T> &a, const U &b)
{
    return a - Fraction<T>(b);
}

template<typename T, typename U>
Fraction<T> operator-(const U &a, const Fraction<T> &b)
{
    return Fraction<T>(a) - b;
}

template<typename T>
Fraction<T> operator*(const Fraction<T> &a, const Fraction<T> &b)
{
    auto c = a;
    c *= b;
    return c;
}

template<typename T, typename U>
Fraction<T> operator*(const Fraction<T> &a, const U &b)
{
    return a * Fraction<T>(b);
}

template<typename T, typename U>
Fraction<T> operator*(const U &a, const Fraction<T> &b)
{
    return Fraction<T>(a) * b;
}

template<typename T>
Fraction<T> operator/(const Fraction<T> &a, const Fraction<T> &b)
{
    auto c = a;
    c /= b;
    return c;
}

template<typename T, typename U>
Fraction<T> operator/(const Fraction<T> &a, const U &b)
{
    return a / Fraction<T>(b);
}

template<typename T, typename U>
Fraction<T> operator/(const U &a, const Fraction<T> &b)
{
    return Fraction<T>(a) / b;
}

template<typename T>
Fraction<T> operator%(const Fraction<T> &a, const Fraction<T> &b)
{
    auto c = a;
    c %= b;
    return c;
}

template<typename T, typename U>
Fraction<T> operator%(const Fraction<T> &a, const U &b)
{
    return a % Fraction<T>(b);
}

template<typename T, typename U>
Fraction<T> operator%(const U &a, const Fraction<T> &b)
{
    return Fraction<T>(a) % b;
}

template<typename T>
std::ostream &operator<<(std::ostream &os, const Fraction<T> &v)
{
    os << v.numerator << "/" << v.denominator;
    return os;
}

}} // namespace Aurora::StreamCamera