/*
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#ifndef __STREAMCAMERA_COMPAT__
#define __STREAMCAMERA_COMPAT__

#include <stdbool.h>
#include <stdint.h>
#include <stddef.h>
#include <unistd.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* CameraInfo */

struct sc_camera_info;
typedef struct sc_camera_info * sc_camera_info_t;

typedef enum {
    SC_CAMERA_FACING_UKNOWN,
    SC_CAMERA_FACING_FRONT,
    SC_CAMERA_FACING_REAR,
    SC_CAMERA_FACING_SCREEN,
} sc_camera_facing_t;

struct sc_camera_info_data {
    const char *id;
    const char *name;
    const char *provider;
    sc_camera_facing_t facing;
    int mount_angle;
    const char *metadata;
};

const char *sc_camera_info_get_id(sc_camera_info_t);

bool sc_camera_info_get_data(sc_camera_info_t,
                                        struct sc_camera_info_data *data);

void sc_camera_info_unref(sc_camera_info_t info);

/* CameraCapability */

struct sc_camera_capability_list;
typedef struct sc_camera_capability_list * sc_camera_capability_list_t;

struct sc_camera_capability_data {
    unsigned int width;         ///< Frame width
    unsigned int height;        ///< Frame height
    unsigned int fps;           ///< Frame rate
};

size_t sc_camera_capability_list_size(sc_camera_capability_list_t);

bool
sc_camera_capability_list_item(
        sc_camera_capability_list_t list,
        unsigned int index,
        struct sc_camera_capability_data *data);

void
sc_camera_capability_list_unref(
        sc_camera_capability_list_t);

/* PixelFormatList */

struct sc_pixel_format_list;
typedef struct sc_pixel_format_list * sc_pixel_format_list_t;

typedef enum {
    SC_PIXEL_FORMAT_INVALID = 0,
    SC_PIXEL_FORMAT_I420,
    SC_PIXEL_FORMAT_NV12,
    SC_PIXEL_FORMAT_YCBCR_FLEXIBLE = 0xff,
} sc_pixel_format_t;

size_t
sc_pixel_format_list_size(
        sc_pixel_format_list_t list);

bool
sc_pixel_format_list_item(
        sc_pixel_format_list_t list,
        unsigned int index,
        sc_pixel_format_t *format);

void
sc_pixel_format_list_unref(
        sc_pixel_format_list_t);

/* YCbCrFrame */

struct sc_ycbcr_frame;
typedef struct sc_ycbcr_frame * sc_ycbcr_frame_t;

struct sc_ycbcr_frame_data {
    const uint8_t *y;           ///< Pointer to luma values
    const uint8_t *cb;          ///< Pointer to cromaB values
    const uint8_t *cr;          ///< Pointer to cromaR values
    uint16_t y_stride;          ///< Luma stride
    uint16_t c_stride;          ///< Chroma stride
    uint16_t chroma_step;       ///< A step between chroma values.
                                ///< chromaStep == 1 for planar format
                                ///< chromaStep == 2 for semi-planar formats.
    uint16_t width;             ///< Frame width
    uint16_t height;            ///< Frame height
    uint64_t timestamp_us;      ///< Monotonically increasing timestamp in microseconds
};

void
sc_ycbcr_frame_get_data(
        sc_ycbcr_frame_t frame,
        struct sc_ycbcr_frame_data *data);

sc_ycbcr_frame_t sc_ycbcr_frame_ref(sc_ycbcr_frame_t);

typedef void (*sc_ycbcr_frame_release_cb)(void *cb_data);

sc_ycbcr_frame_t sc_ycbcr_frame_create(
        struct sc_ycbcr_frame_data *data,
        sc_ycbcr_frame_release_cb release_cb,
        void *cb_data);

void sc_ycbcr_frame_unref(sc_ycbcr_frame_t);

/* GraphicBuffer */

struct sc_graphic_buffer;
typedef struct sc_graphic_buffer * sc_graphic_buffer_t;

typedef enum {
    SC_HANDLE_TYPE_NONE,
    SC_HANDLE_TYPE_EGL,
    SC_HANDLE_TYPE_GBM_IMPORT_DATA
} sc_handle_type_t;

struct sc_graphic_buffer_data {
    uint16_t width;                     ///< Picture width
    uint16_t height;                    ///< Picture height

    uint64_t timestamp_us;              ///< Capture timestamp in microseconds

    sc_pixel_format_t pixel_format;     ///< Pixel format

    const void *handle;                 ///< Hardware-specific handle for the underlying media buffer
    sc_handle_type_t handle_type;       ///< Handle type
};

void sc_graphic_buffer_get_data(sc_graphic_buffer_t, struct sc_graphic_buffer_data *data);

/**
 * \brief  Map YUV video frame.
 *
 * The frame should have sc_pixel_format_t <= SC_PIXEL_FORMAT_YCBCR_FLEXIBLE
 *
 * \returns A pointer to data descriptor or nullptr if data cannot be mapped in this way.
 */
sc_ycbcr_frame_t sc_graphic_buffer_map_ycbcr(sc_graphic_buffer_t);

sc_graphic_buffer_t sc_graphic_buffer_ref(sc_graphic_buffer_t);

void sc_graphic_buffer_unref(sc_graphic_buffer_t);

/* CameraListener */

struct sc_camera_listener;
typedef struct sc_camera_listener * sc_camera_listener_t;

typedef enum {
    SC_CAMERA_PARAMETER_FLASH_MODE,
    SC_CAMERA_PARAMETER_INVALID,
} sc_camera_parameter_t;

struct sc_camera_listener_data {
   void (*on_camera_frame)(sc_graphic_buffer_t, void *user_data);

   void (*on_camera_error)(const char *err_desc, void *user_data);

   void (*on_camera_parameter_changed)(
           sc_camera_parameter_t, const char *value, void *user_data);
};

sc_camera_listener_t
sc_camera_listener_alloc(
        const struct sc_camera_listener_data *listener,
        void *user_data);

void sc_camera_listener_free(sc_camera_listener_t);

/* Camera */

struct sc_camera;
typedef struct sc_camera * sc_camera_t;

sc_camera_t sc_camera_ref(sc_camera_t);

void sc_camera_unref(sc_camera_t);

void
sc_camera_set_listener(
        sc_camera_t camera,
        sc_camera_listener_t listener);

sc_camera_info_t sc_camera_get_info(sc_camera_t camera);

sc_pixel_format_list_t
sc_camera_get_supported_pixel_formats(sc_camera_t camera);

bool
sc_camera_start_capture(
        sc_camera_t camera,
        struct sc_camera_capability_data *cap);

bool sc_camera_stop_capture(sc_camera_t camera);

bool sc_camera_capture_started(sc_camera_t camera);

char *
sc_camera_get_parameter_range(
        sc_camera_t camera,
        sc_camera_parameter_t parameter);

char *
sc_camera_get_parameter(
        sc_camera_t camera,
        sc_camera_parameter_t parameter);

bool
sc_camera_set_parameter(
        sc_camera_t camera,
        sc_camera_parameter_t parameter,
        const char *value);

/* CameraManager */

struct sc_camera_manager;
typedef struct sc_camera_manager * sc_camera_manager_t;

sc_camera_manager_t sc_camera_manager_get(void);

int sc_camera_manager_get_number_of_cameras(sc_camera_manager_t manager);

sc_camera_info_t
sc_camera_manager_get_camera_info(
        sc_camera_manager_t manager,
        int number);

sc_camera_capability_list_t
sc_camera_manager_query_capabilities(
        sc_camera_manager_t manager,
        const char *camera_id);

sc_camera_t
sc_camera_manager_open_camera(
        sc_camera_manager_t manager,
        const char *camera_id);

/* VideoEncoderMetadata */

typedef enum {
    SC_CODEC_TYPE_UNKNOWN,
    SC_CODEC_TYPE_VP8,
    SC_CODEC_TYPE_VP9,
    SC_CODEC_TYPE_H264,
    SC_CODEC_TYPE_H265,
} sc_codec_type_t;

typedef enum {
    SC_FRAME_TYPE_KEY,
    SC_FRAME_TYPE_DELTA,
} sc_frame_type_t;

struct sc_video_encoder_metadata {
    sc_codec_type_t codec_type;  ///< Codec type
    int width;                   ///< Input video width
    int height;                  ///< Input video height
    int stride;                  ///< Stride between video lines if input data is padded.
                                 ///< \a stride must be greater or equal to \a width.
    int slice_height;            ///< The actual height of a video frame if input data is padded.
                                 ///< \a sliceHeight must be greater or equal to \a height.
    int bitrate;                 ///< Initial bitrate of the output video stream
    int framerate;               ///< Input video frame rate. Must not be 0
};

/**
 * Video encoder parameter
 */
typedef enum  {
    SC_VENC_PARAM_TYPE,                ///< R/O, "hardware" or "software"
    SC_VENC_PARAM_BITRATE_MODE,        ///< "constant", "variable"
    SC_VENC_PARAM_PREPROCESS_ROTATION, ///< "0", "90", "180", "270"
    SC_VENC_PARAM_PREPROCESS_MIRROR_H, ///< "true", "false"
    SC_VENC_PARAM_PREPROCESS_MIRROR_V, ///< "true", "false"
    SC_VENC_PARAM_H264_PROFILE,        ///< "default" = auto
    SC_VENC_PARAM_H264_LEVEL,          ///< "default" = auto
    SC_VENC_PARAM_H264_IFRAME_INTERVAL_SEC, ///< float "0.1-20.0", "2.0" is default
    SC_VENC_PARAM_SIZE_RANGE,          ///< Video size range supported by the codec
    SC_VENC_PARAM_ALIGNMENT,           ///< Video data alignment required by the codec
    SC_VENC_PARAM_INVALID,             ///< The last element of the enum
} sc_video_encoder_parameter_t;

/* VideoEncoderListener */

struct sc_video_encoder_listener;
typedef struct sc_video_encoder_listener * sc_video_encoder_listener_t;

struct sc_video_encoder_listener_data {
    void (*on_encoded_frame)(
            uint8_t *data,
            size_t size,
            uint64_t timestamp_us,
            sc_frame_type_t frame_type,
            void *user_data);

    void (*on_encoder_parameter_changed)(
            sc_video_encoder_parameter_t param,
            const char *value,
            void *user_data);

    void (*on_encoder_error)(const char *error_desc, void *user_data);

    void (*on_encoder_eos)(void *user_data);
};

sc_video_encoder_listener_t
sc_video_encoder_listener_alloc(
        const struct sc_video_encoder_listener_data *listener,
        void *user_data);

void sc_video_encoder_listener_free(sc_video_encoder_listener_t listener);

/* VideoEncoder */

struct sc_video_encoder;
typedef struct sc_video_encoder * sc_video_encoder_t;

bool
sc_video_encoder_init(
        sc_video_encoder_t encoder,
        struct sc_video_encoder_metadata *metadata);

sc_pixel_format_list_t
sc_video_encoder_get_supported_pixel_formats(
        sc_video_encoder_t encoder);

bool
sc_video_encoder_encode_ycbcr_frame(
        sc_video_encoder_t encoder,
        sc_ycbcr_frame_t frame,
        bool force_sync);

void
sc_video_encoder_set_bitrate(
        sc_video_encoder_t encoder,
        uint32_t bps);

char *
sc_video_encoder_get_parameter_range(
        sc_video_encoder_t encoder,
        sc_video_encoder_parameter_t parameter);

char *
sc_video_encoder_get_parameter(
        sc_video_encoder_t encoder,
        sc_video_encoder_parameter_t parameter);

bool
sc_video_encoder_set_parameter(
        sc_video_encoder_t encoder,
        sc_video_encoder_parameter_t parameter,
        const char *value);

void
sc_video_encoder_set_listener(
        sc_video_encoder_t encoder,
        sc_video_encoder_listener_t listener);

sc_video_encoder_t sc_video_encoder_ref(sc_video_encoder_t encoder);

void sc_video_encoder_unref(sc_video_encoder_t encoder);

/* VideoDecoderListener */

struct sc_video_decoder_listener;
typedef struct sc_video_decoder_listener * sc_video_decoder_listener_t;

/**
 * Video decoder parameter
 */
typedef enum {
    SC_VDEC_PARAM_TYPE,           ///< R/O, "hardware" or "software"
    SC_VDEC_PARAM_CROP_RECTANGLE, ///< R/O "(<top>,<left>,<width>,<height>)" -
                                  ///< crop area in pixels
    SC_VDEC_PARAM_SIZE_RANGE,     ///< Video size range supported by the codec
    SC_VDEC_PARAM_ALIGNMENT,      ///< Video data alignment required by the codec
    SC_VDEC_PARAM_INVALID,        ///< The last element of the enum
} sc_video_decoder_parameter_t;

struct sc_video_decoder_listener_data {
    /**
     * \brief Called when a frame is decoded.
     *
     * Some devices cannot produce graphic buffers, so they call
     * on_decoded_ycbcr_frame(). The buffer is not shareable and the data will
     * become invalid after the application returns from this callback.
     */
    void (*on_decoded_ycbcr_frame)(
            const struct sc_ycbcr_frame_data *frame,
            void *user_data);

    /**
     * \brief Called when a frame is decoded.
     *
     * \param[out] buffer A reference to the graphic buffer with decoded data.<br>
     * The buffer remain valid until the reference is deleted but it shouldn't be
     * held for too long or the decoder's empty buffer queue may be exausted.
     */
    void (*on_decoded_graphic_buffer)(
            sc_graphic_buffer_t buffer,
            void *user_data);

    /// Called when a hardware-initiated change of a parameter occurs.
    void (*on_decoder_parameter_changed)(
            sc_video_decoder_parameter_t parameter,
            const char *value,
            void *user_data);

    /// Called when a decoder error occurs.
    void (*on_decoder_error)(
            const char *error_desc,
            void *user_data);

    /// Called on EOS.
    void (*on_decoder_eos)(void *user_data);
};

sc_video_decoder_listener_t
sc_video_decoder_listener_alloc(
        const struct sc_video_decoder_listener_data *listener,
        void *user_data);

void sc_video_decoder_listener_free(sc_video_decoder_listener_t listener);

/* VideoDecoder */

struct sc_video_decoder;
typedef struct sc_video_decoder * sc_video_decoder_t;

struct sc_video_decoder_metadata {
    sc_codec_type_t codec_type; ///< Codec type
    int width;                  ///< Supposed video width to preallocate video buffers
    int height;                 ///< Supposed video height
    int framerate;              ///< Video frame rate
    void *codec_specific;       ///< Codec-specific data, for example SPS/PPS block for H264
    size_t codec_specific_size; ///< Size of codec-specific data
};

bool
sc_video_decoder_init(
        sc_video_decoder_t decoder,
        const struct sc_video_decoder_metadata *meta);

sc_pixel_format_list_t
sc_video_decoder_get_supported_pixel_formats(
        sc_video_decoder_t decoder);

bool
sc_video_decoder_decode(
        sc_video_decoder_t decoder,
        const uint8_t *data,
        size_t size,
        uint64_t timestamp_us,
        sc_frame_type_t frame_type,
        void (*release_callback)(void *),
        void *release_callback_data);

void sc_video_decoder_flush(sc_video_decoder_t decoder);

void sc_video_decoder_drain(sc_video_decoder_t decoder);

void sc_video_decoder_stop(sc_video_decoder_t decoder);

char *
sc_video_decoder_get_parameter_range(
        sc_video_decoder_t decoder,
        sc_video_decoder_parameter_t parameter);

char *
sc_video_decoder_get_parameter(
        sc_video_decoder_t decoder,
        sc_video_decoder_parameter_t parameter);

bool
sc_video_decoder_set_parameter(
        sc_video_decoder_t decoder,
        sc_video_decoder_parameter_t parameter,
        const char *value);

void
sc_video_decoder_set_listener(
        sc_video_decoder_t decoder,
        sc_video_decoder_listener_t listener);

sc_video_decoder_t sc_video_decoder_ref(sc_video_decoder_t decoder);

void sc_video_decoder_unref(sc_video_decoder_t decoder);

/* VideoCodecInfo */

struct sc_video_codec_info_list;
typedef struct sc_video_codec_info_list * sc_video_codec_info_list_t;

struct sc_video_codec_info_data {
    sc_codec_type_t type;       ///< Codec type
    bool is_encoder;            ///< Equals to true for encoders
    const char *id;             ///< A unique implementation-defined codec id
    int priority;               ///< Contains codec priority calculated from codec's
                                ///< implementation priority and codec rank. The higher
                                ///< is better.
    unsigned int max_instances; ///< Contains a hint for the max number of the supported
                                ///< concurrent codec instances. The actual number may be less
                                ///< as it depends on the available resources at time of use
    bool hardware_accelerated;  ///< Equals to true for hardware accelerated codecs
};

size_t
sc_video_codec_info_list_size(
        sc_video_codec_info_list_t list);

bool
sc_video_codec_info_list_item(
        sc_video_codec_info_list_t list,
        unsigned int index,
        struct sc_video_codec_info_data *data);

void
sc_video_codec_info_list_unref(
        sc_video_codec_info_list_t);

/* CodecManager */

struct sc_codec_manager;
typedef struct sc_codec_manager * sc_codec_manager_t;

sc_codec_manager_t sc_codec_manager_get(void);

bool sc_codec_manager_init(sc_codec_manager_t manager);

bool
sc_codec_manager_video_encoder_available(
        sc_codec_manager_t manager,
        sc_codec_type_t codec_type,
        bool hw_only);

bool
sc_codec_manager_video_decoder_available(
        sc_codec_manager_t manager,
        sc_codec_type_t codec_type,
        bool hw_only);

sc_video_encoder_t
sc_codec_manager_create_video_encoder(
        sc_codec_manager_t manager,
        sc_codec_type_t codec_type,
        bool hw_only);

sc_video_decoder_t
sc_codec_manager_create_video_decoder(
        sc_codec_manager_t manager,
        sc_codec_type_t codec_type,
        bool hw_only);

sc_video_codec_info_list_t
sc_codec_manager_list_codecs(
        sc_codec_manager_t manager);

bool
sc_codec_manager_is_codec_available(
        sc_codec_manager_t manager,
        const char *codec_id);

sc_video_encoder_t
sc_codec_manager_create_video_encoder_by_id(
        sc_codec_manager_t manager,
        const char *codec_id);

sc_video_decoder_t
sc_codec_manager_create_video_decoder_by_id(
        sc_codec_manager_t manager,
        const char *codec_id);

/* VideoStreamMetadata */

struct sc_video_stream_metadata;
typedef struct sc_video_stream_metadata * sc_video_stream_metadata_t;

int
sc_video_stream_metadata_width(
        sc_video_stream_metadata_t meta);

int
sc_video_stream_metadata_height(
        sc_video_stream_metadata_t meta);

int
sc_video_stream_metadata_level(
        sc_video_stream_metadata_t meta);

int
sc_video_stream_metadata_profile(
        sc_video_stream_metadata_t meta);

struct sc_nalu;
typedef struct sc_nalu * sc_nalu_t;

uint8_t * sc_nalu_data(sc_nalu_t nalu);

size_t sc_nalu_size(sc_nalu_t nalu);

void sc_nalu_free(sc_nalu_t nalu);

uint8_t sc_nalu_item(sc_nalu_t nalu, int index);

sc_nalu_t
sc_video_stream_metadata_sps(
        sc_video_stream_metadata_t meta);

sc_nalu_t
sc_video_stream_metadata_pps(
        sc_video_stream_metadata_t meta);

struct sc_nalu_desc
{
    off_t offset;
    size_t size;
};

struct sc_nalu_desc_list;
typedef struct sc_nalu_desc_list * sc_nalu_desc_list_t;

size_t sc_nalu_desc_list_size(sc_nalu_desc_list_t);

struct sc_nalu_desc sc_nalu_desc_list_item(sc_nalu_desc_list_t list, unsigned int index);

/* VideoStreamParserListener */

struct sc_video_stream_parser_listener;
typedef struct sc_video_stream_parser_listener * sc_video_stream_parser_listener_t;

struct sc_video_stream_parser_listener_data {
    void (*on_output_available)(
            uint8_t *data,
            size_t size,
            sc_nalu_desc_list_t nalu_desc_list,
            uint64_t timestamp_us,
            sc_frame_type_t frame_type,
            void *user_data);

    void (*on_stream_changed)(
            sc_video_stream_metadata_t meta, void *user_data);

    void (*on_parser_error)(const char *error_desc, void *user_data);

    void (*on_parser_eos)(void *user_data);
};

sc_video_stream_parser_listener_t
sc_video_stream_parser_listener_alloc(
        const struct sc_video_stream_parser_listener_data *listener,
        void *user_data);

void sc_video_stream_parser_listener_free(sc_video_stream_parser_listener_t listener);

/* VideoStreamParser */

struct sc_video_stream_parser;
typedef struct sc_video_stream_parser * sc_video_stream_parser_t;

typedef enum {
    None
} sc_conversion_type_t;

typedef enum {
    SC_REQUEST_DEFAULT = 0x00,
    SC_REQUEST_COMPLETE_NALU = 0x01,
    SC_REQUEST_COMPLETE_ACCESS_UNIT = 0x10
} sc_request_type_t;

sc_video_stream_parser_t
sc_video_stream_parser_create();

void sc_video_stream_parser_unref(sc_video_stream_parser_t);

bool sc_video_stream_parser_parse(
        sc_video_stream_parser_t parser,
        const uint8_t *data,
        size_t size,
        uint64_t timestampUs,
        sc_request_type_t requestType);

struct sc_video_stream_parser_settings;
typedef struct sc_video_stream_parser_settings * sc_video_stream_parser_settings_t;

void sc_video_stream_parser_apply_settings(
        sc_video_stream_parser_t parser,
        sc_video_stream_parser_settings_t* settings);

void sc_video_stream_parser_settings_unref(sc_video_stream_parser_settings_t);

sc_video_stream_metadata_t sc_video_stream_parser_get_metadata(sc_video_stream_parser_t);

sc_video_stream_metadata_t sc_video_stream_metadata_copy(sc_video_stream_metadata_t src);

sc_video_stream_metadata_t sc_video_stream_metadata_move(sc_video_stream_metadata_t *src);

void sc_video_stream_metadata_free(sc_video_stream_metadata_t meta);

void sc_video_stream_parser_set_listener(
        sc_video_stream_parser_t parser,
        sc_video_stream_parser_listener_t listener);

/* ParserSettingsBuilder */

struct sc_parser_settings_builder;
typedef struct sc_parser_settings_builder * sc_parser_settings_builder_t;

sc_parser_settings_builder_t sc_parser_settings_builder_create();

void sc_parser_settings_builder_unref(sc_parser_settings_builder_t);

sc_video_stream_parser_settings_t sc_parser_settings_builder_build(sc_parser_settings_builder_t);

#ifdef __cplusplus
} /* extern "C" */
#endif /* __cplusplus */

#endif /* __STREAMCAMERA_COMPAT__ */
/* vim: set ts=4 et sw=4 tw=80: */
