/*
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#ifndef __STREAMCAMERA_CODEC__
#define __STREAMCAMERA_CODEC__

/**
 * \file streamcamera-codec.h
 * Video codec support and data types.
 */

#include <sys/types.h>

#include <string>
#include <vector>
#include <memory>
#include <mutex>

#include "streamcamera.h"

namespace Aurora {
namespace StreamCamera {

/// Video codec type.
enum class CodecType
{
    Unknown,
    VP8,
    VP9,
    H264,
    H265
};

/// Video frame type.
enum class FrameType
{
    Key,
    Delta,
};

/// Video encoder settings.
struct VideoEncoderMetadata
{
    CodecType codecType;         ///< Codec type
    int width;                   ///< Input video width
    int height;                  ///< Input video height
    int stride;                  ///< Stride between video lines if input data is padded.
                                 ///< \a stride must be greater or equal to \a width.
    int sliceHeight;             ///< The actual height of a video frame if input data is padded.
                                 ///< \a sliceHeight must be greater or equal to \a height.
    int bitrate;                 ///< Initial bitrate of the output video stream
    int framerate;               ///< Input video frame rate. Must not be 0
};

/**
 * Basic information about the codec.
 * \since v1.2.0
 */
struct VideoCodecInfo
{
    CodecType type;             ///< Codec type
    bool isEncoder;             ///< Equals to true for encoders
    std::string id;             ///< A unique implementation-defined codec id
    int priority;               ///< Contains codec priority calculated from codec's
                                ///< implementation priority and codec rank. The higher
                                ///< is better.
    unsigned int maxInstances;  ///< Contains a hint for the max number of the supported
                                ///< concurrent codec instances. The actual number may be less
                                ///< as it depends on the available resources at time of use
    bool hardwareAccelerated;   ///< Equals to true for hardware accelerated codecs
};

/**
 * Extended information about the codec.
 * \since v1.5.0
 */
struct VideoCodecExtendedInfo
{
    virtual SizeRange getSizeRange() const = 0; ///< Get video size range supported by the codec
    virtual Size getAlignment() const = 0;      ///< Get video data alignment required by the codec
    virtual ~VideoCodecExtendedInfo() = default;
};

/**
 * Video encoder parameter
 */
enum class VideoEncoderParameter : unsigned int
{
    Type,                         ///< R/O, "hardware" or "software"
    BitRateMode,                  ///< "constant", "variable"
    PreprocessRotation,           ///< "0", "90", "180", "270"
    PreprocessMirrorH,            ///< "true", "false"
    PreprocessMirrorV,            ///< "true", "false"
    H264Profile,                  ///< "default" = auto
    H264Level,                    ///< "default" = auto
    H264IFrameIntervalSec,        ///< float "0.1-20.0", "2.0" is default
    SizeRange,                    ///< Picture size range the codec supports
                                  ///< "(minw,minh)-(maxw,maxh)", \see SizeRange
    Alignment,                    ///< Picture data alignment requirements
                                  ///< "(w,h)" \see Size
    Last,                         ///< A helper to iterate parameters
    Invalid = Last,               ///< Deprecated, do not use.
};

/**
 * \brief Callbacks for the video encoder.
 *
 * The user must implement this class and provide callbacks for the encoder
 * with VideoEncoder::lockAndSetListener.
 */
class VideoEncoderListener
{
public:
    virtual ~VideoEncoderListener() = default;

    /**
     * \brief Called when a buffer encoding is completed.
     *
     * Called on a different thread. The time spent in this function should be minimized.
     *
     * \param[out] data Pointer to encoded data
     * \param[out] size Encoded data size
     * \param[out] timestampUs Video frame timestamp
     * \param[out] frameType Indicates whether it is a key frame
     */
    virtual void onEncodedFrame(uint8_t *data,
                                size_t size,
                                uint64_t timestampUs,
                                FrameType frameType) = 0;

    /// Called when a hardware-initiated change of a parameter occurs. For future purposes.
    virtual void onEncoderParameterChanged(VideoEncoderParameter, const std::string &value) = 0;

    /// Called when an encoder error occurs.
    virtual void onEncoderError(const std::string &errorDescription) = 0;

    /// Called when an end-of-stream occurs.
    virtual void onEncoderEOS() = 0;
};

/**
 * \brief Video encoder
 */
class VideoEncoder
{
public:
    virtual ~VideoEncoder() = default;

    /**
     * \brief Initialize the video encoder.
     *
     * After calling this function, the encoder is able to consume raw video frames.
     * Subsequent calls VideoEncoder::init() will fail. Destroy the encoder object
     * to stop the video encoder and generate EOS.
     *
     * \returns True if the video encoder is successfully initialized.
     */
    virtual bool init(VideoEncoderMetadata metadata) = 0;

    /**
     * \brief List supported pixel formats.
     *
     * The list is sorted by codec's preference.
     *
     * \returns A list of supported pixel formats, or an empty list if an error occurs.
     */
    virtual std::vector<PixelFormat> getSupportedPixelFormats() = 0;

    /**
     * \brief Start encoding of the video frame.
     *
     * The encoder does not necessarily return the encoded frame immediately. Some encoders
     * should be fed by new video frames. The encoder should be destroyed at the EOS.
     *
     * \param[in] frame The video frame to encode
     * \param[in] forceSync Ask encoder to generate key frame
     * \returns True if the frame is accepted for encoding.
     */
    virtual bool encode(std::shared_ptr<const StreamCamera::YCbCrFrame> frame,
                        bool forceSync) = 0;

    virtual bool encode(std::shared_ptr<const StreamCamera::RawImageFrame> frame,
                        bool forceSync) = 0;

    /**
     * \brief Set encoder bit rate.
     *
     * Can be called in any time after VideoEncoder:init() was called. It is useful in CBR
     * mode when an external bitrate control algorithm is used.
     */
    virtual void setBitrate(uint32_t bps) = 0;

    /**
     * \brief Get possible values for a parameter.
     *
     * \param[in] param Parameter index
     * \returns Return value syntax:<br>
     *   * "v1,v2,v3" for a possible set of values, for example "1,2" or "false,true"
     *   * "v1-v10" for a сontinuous value, for example "1-10" or "0.0-1.0" for float values
     *   * "-"  for a read-only value
     *   * "x" for a rectangle (the value is "(x,y,w,h)")
     *   * empty string if an error occurs
     *
     * \see Camera::getParameterRange
     */
    virtual std::string getParameterRange(VideoEncoderParameter param) const = 0;

    /**
     * \brief Get current value for a parameter.
     *
     * \param[in] param Parameter index
     * \returns Return value syntax for the getParameter():<br>
     *   * "v"
     *   * "(x,y,w,h)" for a rectangle
     *   * empty string if an error occurs
     */
    virtual std::string getParameter(VideoEncoderParameter param) const = 0;

    /**
     * \brief Set value of a parameter.
     *
     * \param[in] param Parameter index
     * \param[in] value
     *   * "v" for a discrete or continuous value
     *   * "(x,y,w,h)" for a rectangle
     * \returns True if the value is accepted.
     */
    virtual bool setParameter(VideoEncoderParameter param, const std::string &value) = 0;

    /**
     * \brief Set callbacks.
     * \deprecated Do not use in newly created code. Use VideoEncoder::lockAndSetListener instead.
     * \param[in] listener An object with callbacks.
     */
    [[deprecated("Use VideoEncoder::lockAndSetListener.")]]
    void setListener(VideoEncoderListener *listener)
    {
        m_encoderListener = listener;
    }

    /**
     * \brief Set callbacks. A thread-safe version of the setListener function.
     * 
     * Guarantees only that calling callbacks and changing listeners are mutually exclusive.
     * 
     * \param[in] listener An object with callbacks. Use nullptr to unset before destroying a listener.
     */
    void lockAndSetListener(VideoEncoderListener *listener);

    /**
     * \brief Get basic information about the codec.
     * \returns VideoCodecInfo
     * \since v1.5.0
     */
    virtual const VideoCodecInfo &getInfo() const = 0;

    /**
     * \brief Get extended codec information.
     * The VideoCodecExtendedInfo may hold a reference to the codec.
     * \returns VideoCodecExtendedInfo
     * \since v1.5.0
     */
    virtual std::shared_ptr<VideoCodecExtendedInfo> getExtendedInfo() const = 0;

     /**
     * \brief Check if the encoder supports video stream with given parameters.
     *
     * \param[in] meta Video stream parameters. Set values to 0 if they do not
     * matter
     * \returns True if the decoder supports video stream with given parameters
     * \since v1.5.0
     */
    virtual bool supports(const VideoEncoderMetadata &meta) const = 0;

    /**
     * \brief Start encoding of the video frame.
     *
     * The encoder is not necessarily returns the encoded frame immediately. Some encoders
     * should be fed by new video frames. The encoder should be destroyed at the EOS.
     *
     * \param[in] frame The video frame to encode
     * \param[in] forceSync Ask encoder to generate key frame
     * \returns True if the frame is accepted for encoding.
     */
    virtual bool encode(std::shared_ptr<const StreamCamera::Frame> frame,
                        bool forceSync) = 0;

    /**
     * \brief Inform the encoder about EOS.
     *
     * Wait for onEncoderEOS or onEncoderError callback to ensure that all frames in
     * the queue have been encoded. After this call, the encoder is unavailable. You
     * may not need to call eos() if the frames in the queue are not needed, in which
     * case just call the destructor of the encoder instance.
     *
     * \since v1.7.5
     * \see VideoEncoderListener::onEncoderEOS
     * \see VideoEncoderListener::onEncoderError
     */
    virtual void eos() = 0;

protected:
    VideoEncoderListener *m_encoderListener = nullptr;
    mutable std::recursive_mutex m_encoderListenerMutex;
};

/**
 * Video decoder parameter
 */
enum class VideoDecoderParameter : unsigned int
{
    Type,               ///< R/O "hardware" or "software" - codec type.
    CropRectangle,      ///< R/O "(<top>,<left>,<width>,<height>)" -
                        ///< crop area in pixels, \see Rectangle
    SizeRange,          ///< Pcture size range the codec supports
                        ///< "(minw,minh)-(maxw,maxh)", \see SizeRange
    Alignment,          ///< Picture data alignment requirements
                        ///< "(w,h)" \see Size
    Last,               ///< A helper to iterate parameters
    Invalid = Last,     ///< Deprecated, do not use.
};

/**
 * \brief Callbacks for the video decoder.
 *
 * The user must implement this class and provide callbacks for the decoder
 * with VideoDecoder::lockAndSetListener.
 */
class VideoDecoderListener
{
public:
    virtual ~VideoDecoderListener() = default;

    /**
     * \brief Called when a frame is decoded.
     *
     * Some devices cannot produce graphic buffers, so they call
     * onDecodedYCbCrFrame(). The buffer is not shareable and the data will
     * become invalid after the application returns from this callback.
     */
    virtual void onDecodedYCbCrFrame(const StreamCamera::YCbCrFrame *frame) = 0;

    /**
     * \brief Called when a frame is decoded.
     *
     * \param[out] buffer A reference to the graphic buffer with decoded data.<br>
     * The buffer remain valid until the reference is deleted but it shouldn't be
     * held for too long or the decoder's empty buffer queue may be exausted.
     */
    virtual void onDecodedGraphicBuffer(std::shared_ptr<StreamCamera::GraphicBuffer> buffer) = 0;

    /// Called when a hardware-initiated change of a parameter occurs.
    virtual void onDecoderParameterChanged(VideoDecoderParameter, const std::string &value) = 0;

    /// Called when a decoder error occurs.
    virtual void onDecoderError(const std::string &errorDescription) = 0;

    /// Called on EOS.
    virtual void onDecoderEOS() = 0;
};

/**
 * Video decoder settings
 */
struct VideoDecoderMetadata
{
    CodecType codecType;            ///< Codec type
    int width;                      ///< Supposed video width to preallocate video buffers
    int height;                     ///< Supposed video height
    int framerate;                  ///< Video frame rate
    void *codecSpecific = nullptr;  ///< Codec-specific data, for example SPS/PPS block for H264
    size_t codecSpecificSize = 0;   ///< Size of codec-specific data
};

/**
 * Video decoder
 */
class VideoDecoder
{
public:
    virtual ~VideoDecoder() = default;

    /**
     * \brief Initialize the video encoder.
     *
     * After this function is called, the decoder is able to consume encoded video
     * frames. Subsequent calls VideoDecoder::init() will fail. Destroy the decoder
     * object to stop the video decoder and generate EOS.
     *
     * \returns True if the video decoder is succesfully initialized.
     */
    virtual bool init(VideoDecoderMetadata metadata) = 0;

    /**
     * \brief List supported pixel formats.
     *
     * The list is sorted by codec's preference.
     *
     * \returns A list of supported pixel formats, or an empty list if an error occurs.
     */
    virtual std::vector<PixelFormat> getSupportedPixelFormats() = 0;

    /**
     * \brief Decode video frame.
     *
     * May block if the codec queue is full, so may be called from a separate thread.
     *
     * \param[in] data A pointer to encoded video unit
     * \param[in] size Encoded video unit size
     * \param[in] timestampUs A video frame timestamp
     * \param[in] releaseCallback This function is called when the data memory is no longer needed
     * \param[in] releaseCallbackData releaseCallback parameter
     */
    virtual bool decode(const uint8_t *data,
                        size_t size,
                        uint64_t timestampUs,
                        FrameType frameType,
                        void (*releaseCallback)(void *),
                        void *releaseCallbackData) = 0;

    /// Flush the video decoder queue.
    virtual void flush() = 0;

    /// Drop the video decoder queue.
    virtual void drain() = 0;

    /// Stop the decoder.
    virtual void stop() = 0;

    /**
     * \brief Get possible values for a parameter.
     *
     * \param[in] param Parameter index
     * \returns Return value syntax:<br>
     *   * "v1,v2,v3" for a possible set of values, for example "1,2" or "false,true"
     *   * "v1-v10" for a сontinuous value, for example "1-10" or "0.0-1.0" for float values
     *   * "v1,v2-v10" for a discrete value
     *   * "-"  for a read-only value
     *   * "x" for a rectangle (the value is "(x,y,w,h)")
     *   * empty string if an error occurs
     *
     * \see Camera::getParameterRange
     */
    virtual std::string getParameterRange(VideoDecoderParameter param) const = 0;

    /**
     * \brief Get current value for a parameter.
     *
     * \param[in] param Parameter index
     * \returns Return value syntax for the getParameter():<br>
     *   * "v"
     *   * "(x,y,w,h)" for a rectangle
     *   * empty string if an error occurs
     */
    virtual std::string getParameter(VideoDecoderParameter param) const = 0;

    /**
     * \brief Set value of a parameter.
     *
     * \param[in] param Parameter index
     * \param[in] value
     *   * "v" for a discrete or continuous value
     *   * "(x,y,w,h)" for a rectangle
     * \returns True if the value is accepted.
     */
    virtual bool setParameter(VideoDecoderParameter param, const std::string &value) = 0;

    /**
     * \brief Set callbacks.
     * \deprecated Do not use in newly created code. Use VideoDecoder::lockAndSetListener instead.
     * \param[in] listener An object with callbacks.
     */
    [[deprecated("Use VideoDecoder::lockAndSetListener.")]]
    void setListener(VideoDecoderListener *listener)
    {
        m_decoderListener = listener;
    }

    /**
     * \brief Set callbacks. A thread-safe version of the setListener function.
     *
     * Guarantees only that calling callbacks and changing listeners are mutually exclusive.
     *
     * \param[in] listener An object with callbacks. Use nullptr to unset before destroying a listener.
     */
    void lockAndSetListener(VideoDecoderListener *listener);

    /**
     * \brief Get basic information about the codec.
     * \returns VideoCodecInfo
     * \since v1.5.0
     */
    virtual const VideoCodecInfo &getInfo() const = 0;

    /**
     * \brief Get extended codec information.
     * The VideoCodecExtendedInfo may hold a reference to the codec.
     * \returns VideoCodecExtendedInfo
     * \since v1.5.0
     */
    virtual std::shared_ptr<VideoCodecExtendedInfo> getExtendedInfo() const = 0;

     /**
     * \brief Check if the decoder supports video stream with given parameters.
     *
     * \param[in] meta Video stream parameters. Set values to 0 if they do not
     * matter
     * \returns True if the decoder supports video stream with given parameters
     * \since v1.5.0
     */
    virtual bool supports(const VideoDecoderMetadata &meta) const = 0;

protected:
    VideoDecoderListener *m_decoderListener = nullptr;
    mutable std::recursive_mutex m_decoderListenerMutex;
};

/**
 * Codec manager
 */
class CodecManager
{
public:
    virtual ~CodecManager() = default;

    /**
     * \brief (Re-)initialize the codec manager.
     *
     * This method can be used to search for new codecs. Not used for now.
     */
    virtual bool init() = 0;

    /**
     * \brief Check if the encoder is supported.
     *
     * May return false if all suitable codecs are busy.
     *
     * \param[in] codecType Codec type
     * \param[in] hwOnly Do not use software codecs
     * \returns True if the encoder is supported
     */
    virtual bool videoEncoderAvailable(CodecType codecType, bool hwOnly = true) = 0;

     /**
     * \brief Check if the decoder is supported.
     *
     * May return false if all suitable codecs are busy.
     *
     * \param[in] codecType Codec type
     * \param[in] hwOnly Do not use software codecs
     * \returns True if the decoder is supported
     */
    virtual bool videoDecoderAvailable(CodecType codecType, bool hwOnly = true) = 0;

    /**
     * \brief Create video encoder.
     *
     * Returns the highest-priority available encoder for selected \p codecType.
     * May return nullptr if all suitable codecs are busy.
     *
     * \param[in] codecType Codec type
     * \param[in] hwOnly Do not use software codecs
     * \returns A reference to a newly allocated codec
     */
    virtual std::shared_ptr<VideoEncoder> createVideoEncoder(CodecType codecType, bool hwOnly = true) = 0;

    /**
     * \brief Create video decoder.
     *
     * Returns the highest-priority available decoder for selected \p codecType.
     * May return nullptr if all suitable codecs are busy.
     *
     * \param[in] codecType Codec type
     * \param[in] hwOnly Do not use software codecs
     * \returns A reference to a newly allocated codec
     */
    virtual std::shared_ptr<VideoDecoder> createVideoDecoder(CodecType codecType, bool hwOnly = true) = 0;

    /**
     * \brief List available codecs.
     * \return A list of supported codecs sorted by priority in descending
     * order. 
     * \since v1.2.0
     */
    virtual std::vector<VideoCodecInfo> listCodecs() const = 0;

    /**
     * \brief Checks if the codec id is available.
     *
     * \param[in] id Codec id
     * \return False if there are no codec instances available.
     * \since v1.2.0
     */
    virtual bool isCodecIdAvailable(const std::string &id) const = 0;

    /**
     * \brief Create video encoder by id.
     *
     * \param[in] id Codec id.
     *
     * \return nullptr if there are no codec instances available.
     * \since v1.2.0
     */
    virtual std::shared_ptr<VideoEncoder> createVideoEncoderById(const std::string &id) = 0;

    /**
     * \brief Create video decoder by id.
     *
     * \param[in] id Codec id
     * \return nullptr if there are no codec instances available.
     * \since v1.2.0
     */
    virtual std::shared_ptr<VideoDecoder> createVideoDecoderById(const std::string &id) = 0;

     /**
     * \brief Get extended information about the codec.
     *
     * \param[in] id Video codec id
     * \returns VideoCodecExtendedInfo
     * \since v1.5.0
     */
    virtual std::shared_ptr<VideoCodecExtendedInfo> getVideoCodecExtendedInfo(const std::string &codecId) = 0;

     /**
     * \brief Check if the decoder supports video stream with given parameters.
     *
     * May return false if all suitable codecs are busy.
     *
     * \param[in] id Video decoder id. Pass an empty string to walk through all
     * available decoders. The function returns true if a suitable codec is
     * found.
     * \param[in] meta Video stream parameters. Set values to 0 if they do not
     * matter
     * \param[in] hwOnly Do not use software codecs
     * \returns True if the decoder supports video stream with given parameters
     * \since v1.5.0
     */
    virtual bool videoDecoderSupports(const std::string &id, const VideoDecoderMetadata &meta, bool hwOnly = true) = 0;

     /**
     * \brief Check if the encoder supports video stream with given parameters.
     *
     * May return false if all suitable codecs are busy.
     *
     * \param[in] id Video encoder id. Pass an empty string to walk through all
     * available encoders. The function returns true if a suitable codec is
     * found.
     * \param[in] meta Video stream parameters. Set values to 0 if they do not
     * matter
     * \param[in] hwOnly Do not use software codecs
     * \returns True if the decoder supports video stream with given parameters
     * \since v1.5.0
     */
    virtual bool videoEncoderSupports(const std::string &id, const VideoEncoderMetadata &meta, bool hwOnly = true) = 0;
};

} // namespace StreamCamera
} // namespace Aurora

/**
 * \brief Constructor for the root CodecManager.
 *
 * \returns A pointer to statically allocated and initialized CodecManager.
 */
extern "C" __attribute__((visibility("default")))
Aurora::StreamCamera::CodecManager *StreamCameraCodecManager(void);

#endif // __STREAMCAMERA_CODEC__
/* vim: set ts=4 et sw=4 tw=80: */
