/****************************************************************************
**
** Copyright (C) 2023 Open Mobile Platform LLC.
**
** This file is a part of the QtVideoChat module.
**
** $QT_BEGIN_LICENSE:LGPL21$
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
** $QT_END_LICENSE$
**
****************************************************************************/

#pragma once

#include <QObject>

#include <qmediaobject.h>
#include <qmediabindableinterface.h>
#include <qmediaencodersettings.h>
#include <qmediaenumdebug.h>
#include <qvideosurfaceformat.h>
#include <qvideoframe.h>

#include "qvcencodedvideoframe.h"
#include "qvcsource.h"

#include <qvcglobal.h>

QT_BEGIN_NAMESPACE

class QVCEncoderPrivate;

class Q_VIDEOCHAT_EXPORT QVCEncoder
    : public QMediaObject
    , public QMediaBindableInterface
    , public QVCSource
{
    Q_OBJECT

    Q_INTERFACES(QMediaBindableInterface)
    Q_INTERFACES(QVCSource)

    Q_ENUMS(Status)
    Q_ENUMS(Error)
    Q_PROPERTY(QVCEncoder::Status status READ status NOTIFY statusChanged)
    Q_PROPERTY(QVCEncoder::Error error READ error NOTIFY errorChanged);

public:
    enum Status {
        UnavailableStatus,
        UnloadedStatus,
        LoadingStatus,
        LoadedStatus,
        StoppingStatus,
        StartingStatus,
        StartedStatus
    };

    enum Error
    {
        NoError,
        ResourceError,
        FormatError,
        ConsumerError,
        HardwareError
    };

    explicit QVCEncoder(QObject *parent = nullptr);
    ~QVCEncoder();

    Status status() const;

    Error error() const;
    QString errorString() const;

    QVideoEncoderSettings videoSettings() const;
    void setVideoSettings(const QVideoEncoderSettings &settings);

    QStringList supportedCodecs(bool hardwareOnly = true) const;
    QList<qreal> supportedFrameRates(const QVideoEncoderSettings &settings, bool *continuous) const;
    QList<QSize> supportedResolutions(const QVideoEncoderSettings &settings, bool *continuous) const;
    QList<QVideoFrame::PixelFormat> supportedPixelFormats() const;

    QMediaObject *mediaObject() const override;
    bool setMediaObject(QMediaObject *) override;

    bool start(const QVideoSurfaceFormat &format);
    bool encode(const QVideoFrame &frame);
    void drain();
    void flush();
    void stop();

    bool addSink(QVCSink *sink) override;
    bool removeSink(QVCSink *sink) override;

Q_SIGNALS:
    void encodedVideoFrame(QVCEncodedVideoFrame);
    void statusChanged(QVCEncoder::Status status);
    void errorChanged(QVCEncoder::Error error);

private:
    Q_DISABLE_COPY(QVCEncoder)
    Q_DECLARE_PRIVATE(QVCEncoder)
    Q_PRIVATE_SLOT(d_func(), void _q_error(QVCEncoder::Error, const QString &))
};

QT_END_NAMESPACE

Q_DECLARE_METATYPE(QVCEncoder::Status)
Q_DECLARE_METATYPE(QVCEncoder::Error)

Q_MEDIA_ENUM_DEBUG(QVCEncoder, Status)
Q_MEDIA_ENUM_DEBUG(QVCEncoder, Error)
