/****************************************************************************
**
** Copyright (C) 2023 Open Mobile Platform LLC.
**
** This file is a part of the QtVideoChat module.
**
** $QT_BEGIN_LICENSE:LGPL21$
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
** $QT_END_LICENSE$
**
****************************************************************************/

#pragma once

#include <qobject.h>

#include <qmediaobject.h>
#include <qmediabindableinterface.h>
#include <qvideoframe.h>
#include <qvideosurfaceformat.h>
#include <qmediaencodersettings.h>
#include <qmediaenumdebug.h>

#include <qvcencodedvideoframe.h>

class QAbstractVideoSurface;

#include <qvcglobal.h>

QT_BEGIN_NAMESPACE

class QVCDecoderPrivate;
class Q_VIDEOCHAT_EXPORT QVCDecoder
    : public QMediaObject
    , public QMediaBindableInterface
{
    Q_OBJECT

    Q_INTERFACES(QMediaBindableInterface)

    Q_ENUMS(Status)
    Q_ENUMS(Error)
    Q_PROPERTY(QVCDecoder::Status status READ status NOTIFY statusChanged)
    Q_PROPERTY(QVCDecoder::Error error READ error NOTIFY errorChanged)

public:
    enum Status {
        UnavailableStatus,
        UnloadedStatus,
        LoadingStatus,
        UnloadingStatus,
        LoadedStatus,
        EndOfStreamStatus,
        StoppingStatus,
        StartingStatus,
        StartedStatus
    };

    enum Error
    {
        NoError,
        ResourceError,
        FormatError,
        HardwareError
    };

    explicit QVCDecoder(QObject *parent = nullptr);
    ~QVCDecoder();

    QMediaObject *mediaObject() const override;
    bool setMediaObject(QMediaObject *object) override;

    Status status() const;

    Error error() const;
    QString errorString() const;

    QVideoSurfaceFormat outputFormat();

    QStringList supportedCodecs(bool hardwareOnly = true) const;
    bool isFormatSupported(const QVideoEncoderSettings &videoSettings) const;
    void setVideoSurface(QAbstractVideoSurface *surface);

    bool start(const QVideoEncoderSettings &videoSettings);
    bool decode(const QVCEncodedVideoFrame &frame);
    void drain();
    void flush();
    void stop();

Q_SIGNALS:
    void statusChanged(QVCDecoder::Status status);
    void errorChanged(QVCDecoder::Error error);
    void outputFormatChanged(const QVideoSurfaceFormat &);
    void endOfStream();

protected:
    explicit QVCDecoder(QVCDecoderPrivate &, QObject *parent = nullptr);

private:
    Q_DISABLE_COPY(QVCDecoder)
    Q_DECLARE_PRIVATE(QVCDecoder)
    Q_PRIVATE_SLOT(d_func(), void _q_error(QVCDecoder::Error, const QString &))
    Q_PRIVATE_SLOT(d_func(), void _q_outputFormatChanged(const QVideoSurfaceFormat &))
};

QT_END_NAMESPACE

Q_DECLARE_METATYPE(QVCDecoder::Status)
Q_DECLARE_METATYPE(QVCDecoder::Error)

Q_MEDIA_ENUM_DEBUG(QVCDecoder, Status)
Q_MEDIA_ENUM_DEBUG(QVCDecoder, Error)
