/****************************************************************************
**
** Copyright (C) 2023 Open Mobile Platform LLC.
** Copyright (C) 2015 The Qt Company Ltd.
**
** This file is a part of the QtVideoChat module.
**
** $QT_BEGIN_LICENSE:LGPL21$
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
** $QT_END_LICENSE$
**
****************************************************************************/

#pragma once

#include <QtMultimedia/qmediaobject.h>
#include <QtMultimedia/qmediaenumdebug.h>
#include <QCameraExposure>

#include <qvcglobal.h>

QT_BEGIN_NAMESPACE

class QVCCamera;
class QVCCameraExposurePrivate;

class Q_VIDEOCHAT_EXPORT QVCCameraExposure : public QObject
{
    Q_OBJECT
    Q_PROPERTY(qreal aperture READ aperture NOTIFY apertureChanged)
    Q_PROPERTY(qreal shutterSpeed READ shutterSpeed NOTIFY shutterSpeedChanged)
    Q_PROPERTY(int isoSensitivity READ isoSensitivity NOTIFY isoSensitivityChanged)
    Q_PROPERTY(qreal exposureCompensation READ exposureCompensation WRITE setExposureCompensation NOTIFY exposureCompensationChanged)
    Q_PROPERTY(bool flashReady READ isFlashReady NOTIFY flashReady)
    Q_PROPERTY(QCameraExposure::FlashModes flashMode READ flashMode WRITE setFlashMode)
    Q_PROPERTY(QCameraExposure::ExposureMode exposureMode READ exposureMode WRITE setExposureMode)
    Q_PROPERTY(QCameraExposure::MeteringMode meteringMode READ meteringMode WRITE setMeteringMode)

public:

    bool isAvailable() const;

    QCameraExposure::FlashModes flashMode() const;
    bool isFlashModeSupported(QCameraExposure::FlashModes mode) const;
    bool isFlashReady() const;

    QCameraExposure::ExposureMode exposureMode() const;
    bool isExposureModeSupported(QCameraExposure::ExposureMode mode) const;

    qreal exposureCompensation() const;

    QCameraExposure::MeteringMode meteringMode() const;
    bool isMeteringModeSupported(QCameraExposure::MeteringMode mode) const;

    QPointF spotMeteringPoint() const;
    void setSpotMeteringPoint(const QPointF &point);

    int isoSensitivity() const;
    qreal aperture() const;
    qreal shutterSpeed() const;

    int requestedIsoSensitivity() const;
    qreal requestedAperture() const;
    qreal requestedShutterSpeed() const;

    QList<int> supportedIsoSensitivities(bool *continuous = Q_NULLPTR) const;
    QList<qreal> supportedApertures(bool *continuous = Q_NULLPTR) const;
    QList<qreal> supportedShutterSpeeds(bool *continuous = Q_NULLPTR) const;

public Q_SLOTS:
    void setFlashMode(QCameraExposure::FlashModes mode);
    void setExposureMode(QCameraExposure::ExposureMode mode);
    void setMeteringMode(QCameraExposure::MeteringMode mode);

    void setExposureCompensation(qreal ev);

    void setManualIsoSensitivity(int iso);
    void setAutoIsoSensitivity();

    void setManualAperture(qreal aperture);
    void setAutoAperture();

    void setManualShutterSpeed(qreal seconds);
    void setAutoShutterSpeed();

Q_SIGNALS:
    void flashReady(bool);

    void apertureChanged(qreal);
    void apertureRangeChanged();
    void shutterSpeedChanged(qreal);
    void shutterSpeedRangeChanged();
    void isoSensitivityChanged(int);
    void exposureCompensationChanged(qreal);

private:
    friend class QVCCamera;
    friend class QVCCameraPrivate;
    explicit QVCCameraExposure(QVCCamera *parent = Q_NULLPTR);
    virtual ~QVCCameraExposure();

    Q_DISABLE_COPY(QVCCameraExposure)
    Q_DECLARE_PRIVATE(QVCCameraExposure)
    Q_PRIVATE_SLOT(d_func(), void _q_exposureParameterChanged(int))
    Q_PRIVATE_SLOT(d_func(), void _q_exposureParameterRangeChanged(int))
    QVCCameraExposurePrivate *d_ptr;
};

QT_END_NAMESPACE
