/****************************************************************************
**
** Copyright (C) 2023 Open Mobile Platform LLC.
** Copyright (C) 2015 The Qt Company Ltd.
**
** This file is a part of the QtVideoChat module.
**
** $QT_BEGIN_LICENSE:LGPL21$
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
** $QT_END_LICENSE$
**
****************************************************************************/

#pragma once

#include <QtCore/qstringlist.h>
#include <QtCore/qpair.h>
#include <QtCore/qsize.h>
#include <QtCore/qpoint.h>
#include <QtCore/qrect.h>

#include <QtMultimedia/qmediacontrol.h>
#include <QtMultimedia/qmediaobject.h>

#include <QtMultimedia/qmediaservice.h>
#include <QtMultimedia/qcamera.h>
#include <QtMultimedia/qcameraimageprocessing.h>
#include <QtMultimedia/qcameraviewfindersettings.h>

#include <QtMultimedia/qmediaenumdebug.h>

#include "qvccameraexposure.h"
#include "qvccamerafocus.h"

class QAbstractVideoSurface;

#include <qvcglobal.h>

QT_BEGIN_NAMESPACE

class QVCCameraInfo;

class QVCCameraPrivate;
class Q_VIDEOCHAT_EXPORT QVCCamera : public QMediaObject
{
    Q_OBJECT
    Q_PROPERTY(QCamera::State state READ state NOTIFY stateChanged)
    Q_PROPERTY(QCamera::Status status READ status NOTIFY statusChanged)
    Q_PROPERTY(QCamera::CaptureModes captureMode READ captureMode WRITE setCaptureMode NOTIFY captureModeChanged)
    Q_PROPERTY(QCamera::LockStatus lockStatus READ lockStatus NOTIFY lockStatusChanged)

public:

    explicit QVCCamera(QObject *parent = Q_NULLPTR);
    explicit QVCCamera(const QByteArray& deviceName, QObject *parent = Q_NULLPTR);
    explicit QVCCamera(const QVCCameraInfo& cameraInfo, QObject *parent = Q_NULLPTR);
    explicit QVCCamera(QCamera::Position position, QObject *parent = Q_NULLPTR);
    ~QVCCamera();

    QMultimedia::AvailabilityStatus availability() const;

    QCamera::State state() const;
    QCamera::Status status() const;

    QCamera::CaptureModes captureMode() const;
    bool isCaptureModeSupported(QCamera::CaptureModes mode) const;

    QVCCameraExposure *exposure() const;
    QVCCameraFocus *focus() const;
    QCameraImageProcessing *imageProcessing() const;

    void setViewfinder(QAbstractVideoSurface *surface);

    QCameraViewfinderSettings viewfinderSettings() const;
    void setViewfinderSettings(const QCameraViewfinderSettings &settings);

    QList<QCameraViewfinderSettings> supportedViewfinderSettings(
            const QCameraViewfinderSettings &settings = QCameraViewfinderSettings()) const;

    QList<QSize> supportedViewfinderResolutions(
            const QCameraViewfinderSettings &settings = QCameraViewfinderSettings()) const;

    QList<QCamera::FrameRateRange> supportedViewfinderFrameRateRanges(
            const QCameraViewfinderSettings &settings = QCameraViewfinderSettings()) const;

    QList<QVideoFrame::PixelFormat> supportedViewfinderPixelFormats(
            const QCameraViewfinderSettings &settings = QCameraViewfinderSettings()) const;

    QCamera::Error error() const;
    QString errorString() const;

    QCamera::LockTypes supportedLocks() const;
    QCamera::LockTypes requestedLocks() const;

    QCamera::LockStatus lockStatus() const;
    QCamera::LockStatus lockStatus(QCamera::LockType lock) const;

public Q_SLOTS:
    void setCaptureMode(QCamera::CaptureModes mode);

    void load();
    void unload();

    void start();
    void stop();

    void searchAndLock();
    void unlock();

    void searchAndLock(QCamera::LockTypes locks);
    void unlock(QCamera::LockTypes locks);

Q_SIGNALS:
    void stateChanged(QCamera::State);
    void captureModeChanged(QCamera::CaptureModes);
    void statusChanged(QCamera::Status);

    void locked();
    void lockFailed();

    void lockStatusChanged(QCamera::LockStatus, QCamera::LockChangeReason);
    void lockStatusChanged(QCamera::LockType, QCamera::LockStatus, QCamera::LockChangeReason);

    void error(QCamera::Error);

private:
    Q_DISABLE_COPY(QVCCamera)
    Q_DECLARE_PRIVATE(QVCCamera)
    Q_PRIVATE_SLOT(d_func(), void _q_preparePropertyChange(int))
    Q_PRIVATE_SLOT(d_func(), void _q_restartCamera())
    Q_PRIVATE_SLOT(d_func(), void _q_error(int, const QString &))
    Q_PRIVATE_SLOT(d_func(), void _q_updateLockStatus(QCamera::LockType, QCamera::LockStatus, QCamera::LockChangeReason))
    Q_PRIVATE_SLOT(d_func(), void _q_updateState(QCamera::State))
    friend class QVCCameraInfo;
};

QT_END_NAMESPACE
