/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef MARKET_LIB_PACKAGE_FILTER_HPP
#define MARKET_LIB_PACKAGE_FILTER_HPP

#include "globals.hpp"
#include "package.hpp"

#include <functional>

namespace Market {

/**
 * @brief A function that is applied to a @ref Package class instance and returns
 *        true if the packet satisfies the requirements and false otherwise.
 */
using PackageFilter = std::function<bool(const Package &)>;

/**
 * @brief Ready-to-use package filters
 * @namespace Market::PackageFilters
 */
namespace PackageFilters {

/**
 * @brief Create a package filter that only allows packages that have
 *        the given developer certificate subgroup.
 *
 * @param developerCertificateSubgroup Developer certificate subgroup.
 *
 * @return Package filter bound to a given developer certificate subgroup.
 */
MARKET_LIB_EXPORT_SYMBOL
PackageFilter DeveloperCertificateSubgroup(const std::string &developerCertificateSubgroup);

/**
 * @brief Create a package filter that only allows packages that have
 *        the given developer certificate ID.
 *
 * @param developerCertificateID Developer certificate ID.
 *
 * @return Package filter bound to a given developer certificate ID.
 */
MARKET_LIB_EXPORT_SYMBOL
PackageFilter DeveloperCertificateID(const std::string &developerCertificateID);

} /* namespace PackageFilter */
} /* namespace Market */

#endif /* MARKET_LIB_PACKAGE_FILTER_HPP */
