/**
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef MARKET_LIB_DAEMON_CONNECTION_HPP
#define MARKET_LIB_DAEMON_CONNECTION_HPP

#include "event-func.hpp"
#include "event-subscription.hpp"
#include "event-type.hpp"
#include "globals.hpp"
#include "operation-request.hpp"
#include "operation.hpp"
#include "package-filter.hpp"
#include "package.hpp"
#include "result.hpp"

#include <memory>
#include <string>
#include <vector>

/**
 * @brief Classes and functions for working with the market daemon
 * @namespace Market
 */
namespace Market {

class DaemonConnectionPrivate;

/**
 * @brief Class for package management through the market daemon.
 * @class DaemonConnection daemon-connection.hpp <market/daemon-connection.hpp>
 *
 * @note This class does not have a public constructor. To get a connection,
 *       use the @ref GetDaemonConnection() method, which creates and returns
 *       a single connection to the market daemon.
 *
 * @code{.cpp}
 * DaemonConnection daemon = GetDaemonConnection();
 * @endcode
 */
class MARKET_LIB_EXPORT_SYMBOL DaemonConnection
{
public:
    /**
     * @brief Get device architecture name compatible with RPM
     *        package architecture names.
     *
     * @return The device architecture name.
     */
    const std::string &GetArchitecture() const;

    /**
     * @brief Get the operating system version that is installed
     *        on the device.
     *
     * @return The device architecture name.
     */
    const std::string &GetAuroraVersion() const;

    /**
     * @brief Get active operation by operation UUID.
     *
     * @note This method may return an error result if the list of active
     *       operations does not contain an operation with the specified UUID.
     *
     * @param operationUUID Unique ID of the operation.
     *
     * @return @ref Result containing an active operation or error.
     */
    Result<Operation> GetActiveOperation(const OperationUUID &operationUUID) const;

    /**
     * @brief Get a list of operations that are currently not completed.
     *
     * @note Calling this method will retrieve the active operations
     *       that were created by the executable file that uses this
     *       method. This means that when restarting your executable,
     *       you will be able to see the active operations created
     *       before restarting the executable.
     *
     * @return List of active operations.
     */
    std::vector<Operation> GetActiveOperations() const;

    /**
     * @brief Get a list of completed operations.
     *
     * @note Completed operations are taken from the stash, which is cleared
     *       at the first request for completed operations. This means that if
     *       you call this method a second time without restarting the executable
     *       file, you will get an empty list of completed operations.
     *
     * @return List of completed operations.
     */
    std::vector<Operation> GetCompletedOperations() const;

    /**
     * @brief Get installed package by package ID.
     *
     * @note This method may return an error result if the list of installed
     *       packages does not contain a package with the specified ID.
     *
     * @param packageID Package identifier (package name).
     *
     * @return @ref Result containing an installed package or error.
     */
    Result<Package> GetPackage(const std::string &packageID) const;

    /**
     * @brief Get a list of installed packages.
     *
     * @return List of installed packages.
     */
    std::vector<Package> GetPackageList() const;

    /**
     * @brief Getting a list of installed packages that satisfy the filter rules.
     *
     * @note The library provides ready-to-use filters, which can be found
     *       in the @ref PackageFilters namespace:
     *
     * @code
     * DaemonConnection daemon = GetDaemonConnection();
     *
     * std::vector<Package> packages;
     *
     * packages = daemon->GetPackageList(PackageFilters::DeveloperCertificateID("xxx-xxx-xxx"));
     * packages = daemon->GetPackageList(PackageFilters::DeveloperCertificateSubgroup("regular"));
     * @endcode
     *
     * @return Filtered list of installed packages.
     */
    std::vector<Package> GetPackageList(const PackageFilter &filter) const;

    /**
     * @brief Creates an operation on the given @ref OperationRequest and sends
     *        that operation to the processing queue.
     *
     * @note This method may return an error result if a daemon-side error occurred
     *       while creating the operation.
     *
     * @param request Request for operation creation
     *
     * @return @ref Result containing the UUID of the created operation or error.
     */
    Result<OperationUUID> CreateOperation(const OperationRequest &request) const;

    /**
     * @brief Adds an event handler function to the queue of handlers that will
     *        be called one by one when the specified event occurs.
     *
     * The market daemon client will see @ref PackageInstalled, @ref PackageUpgraded
     * and @ref PackageRemoved events that occurred during the launch of an operation
     * by another market daemon client.
     *
     * The client of the daemon marketplace will see @ref OperationStatusChanged events
     * only for those operations that were created by this client.
     *
     * @parblock
     * @note Currently, after reconnecting the client to the daemon, for example, when
     *       restarting the executable file, it is not possible to track operations
     *       created at the previous startup by subscribing to the @ref OperationStatusChanged
     *       signal.
     * @endparblock
     *
     * @parblock
     * @note For package install, upgrade and remove events, you must use a callback
     *       of type @ref OnPackageEventFunc. For events related to the change of operation
     *       status, you must use a callback of @ref OnOperationEventFunc type.
     * @endparblock
     *
     * @parblock
     * @note If the type of the callback does not match the type of the event, the
     *       std::logic_error exception will be raised.
     * @endparblock
     *
     * @parblock
     * @note The @ref EventSubscription class automatically unsubscribes the market
     *       daemon client from event bound to this event subsription when the destructor
     *       is called for memory safety purposes. This means that the client needs to
     *       save this subscription somewhere.
     * @endparblock
     *
     * @parblock
     * @note Currently, the client can listen events only if the client application uses
     *       [GMainLoop](https://docs.gtk.org/glib/struct.MainLoop.html), for example an
     *       application based on the Qt framework.
     * @endparblock
     *
     * @param eventType Type of event
     * @param func Callback that will be called when an event occurs.
     *
     * @code{.cpp}
     * DaemonConnection daemon = GetDaemonConnection();
     *
     * std::vector<EventSubscription> subscriptions = {
     *     daemon.SubscribeToEvent(EventType::PackageInstalled, [](const Package &package) { ... }),
     *     daemon.SubscribeToEvent(EventType::OperationStatusChanged, [](const Operation &operation) { ... }),
     * };
     * @endcode
     *
     * @return Instance of the @ref EventSubscription class that manages the created
     *         event subscription.
     */
    EventSubscription SubscribeToEvent(EventType eventType, const OnEventFunc &func) const;

    /**
     * @brief Unsubscribe from the event.
     *
     * @note Nothing happens if you pass to this method an instance
     *       of the @ref EventSubscription class, which has already
     *       been unsubscribed before.
     *
     * @param subscription Subscription to an event to be unsubscribed.
     */
    void UnsubscribeFromEvent(const EventSubscription &subscription);

    /**
     * @brief Start the application.
     *
     * @note If application was running in the background
     *       it brings up to the foreground.
     *
     * @param packageID packageID of the application to be started.
     */
    Result<void> Start(const std::string &packageID) const;

    /**
     * @brief Restart the application.
     *
     * @note If application was running in the background
     *       it brings up to the foreground.
     *
     * @param packageID packageID of the application to be restarted.
     */
    Result<void> Restart(const std::string &packageID) const;

    /**
     * @brief Check if the connection to the daemon was successful.
     *
     * @note If you use methods of unsuccessful connection to the daemon,
     *       the exception std::logic_error() will be raised.
     *
     * @code{.cpp}
     * DaemonConnection daemon = GetDaemonConnection();
     *
     * if (!daemon) {
     *      std::cout << "Bad market daemon connection" << std::endl;
     * }
     * @endcode
     */
    explicit operator bool() const;

private:
    friend DaemonConnection GetDaemonConnection();
    DaemonConnection();

    std::shared_ptr<DaemonConnectionPrivate> m_pimpl;
};

/**
 * @brief Get a single connection to the market daemon.
 *
 * @note The application that uses this method should check if the connection
 *       is successful. If you use methods of unsuccessful connection to the
 *       daemon, the exception std::logic_error() will be raised.
 *
 * @code{.cpp}
 * DaemonConnection daemon = GetDaemonConnection();
 *
 * if (!daemon) {
 *      std::cout << "Bad market daemon connection" << std::endl;
 * }
 * @endcode
 */
MARKET_LIB_EXPORT_SYMBOL
DaemonConnection GetDaemonConnection();

} /* namespace Market */

#endif /* MARKET_LIB_DAEMON_CONNECTION_HPP */
