/*
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QJSValue>
#include <QString>

class QObject;

namespace Aurora {
namespace WebView {

class IRendering;
class IInputExtension;
class ITextExtension;
class IContextMenuExtension;
class ILoadRequestExtension;
class IPopupExtension;
class IPdfPrint;
class IFaviconExtension;
class IMediaPlaybackExtension;
class IOverscrollExtension;
class IConnectionInfoExtension;
class IJavaScriptExtension;
class IPasswordManagerExtension;

class IWebView
{
public:
    virtual ~IWebView() = default;

    virtual void reload() = 0;
    /// stop loading a URL if loading
    virtual void stopLoad() = 0;
    /// continue rendering
    virtual void startRender() = 0;
    /// pause rendering (image will become still, device resource usage will be decreased)
    virtual void stopRender() = 0;
    /// @brief can be used to set current device orientation (e.g., for camera to stream correctly)
    /// @param orientation
    virtual void setOrientation(int orientation) = 0;

    virtual void updateSize(const QRectF rect) = 0;
    /**
     * @brief search for a text on a webpage
     *
     * @param text text to search for
     * @param forward indicates whether to search forward or backward within the page
     * @param matchCase indicates whether the search should be case-sensitive
     * @param findNext indicates whether to move to the next found element and highlight it or not
     *
     * @link searchonpage.qml Usage Example @endlink
     */
    virtual void find(const QString &text, bool forward, bool matchCase, bool findNext) = 0;
    /**
     * @brief cancel all searches that are currently active
     *
     * @param clearSelection clear current selection
     */
    virtual void stopFinding(bool clearSelection) = 0;
    /// returns true if the WebView can navigate backwards
    virtual bool canGoBack() = 0;
    /// navigate backwards to the previous page in the WebView history
    virtual void goBack() = 0;
    /// returns true if the WebView can navigate forwards
    virtual bool canGoForward() = 0;
    /// navigate forwards to the next page in the WebView history
    virtual void goForward() = 0;

    // properties
    /// unique identifier
    virtual int id() const = 0;
    /// true if the WebView is valid
    virtual bool isValid() const = 0;
    /// currently loaded URL
    virtual QString url() const = 0;
    /// URL property setter
    virtual void setUrl(const QString &url) = 0;
    /// current title of the loaded web-page
    virtual QString title() const = 0;
    /// check if WebView instance is loading a web-page
    virtual bool loading() const = 0;
    /// current loading progress, ranges from 0.0 to 1.0
    virtual double loadingProgress() const = 0;
    /// User-Agent
    virtual QString userAgent() const = 0;
    /// set the User-Agent. Page reload is required for new value to be applied.
    ///
    /// Empty string disables override mode and sets the default value.
    virtual void setUserAgent(const QString &userAgent) = 0;
    /// Device Pixel Ratio
    virtual qreal DPR() const = 0;
    /// set the DPR. Reload to apply the new value.
    virtual void setDPR(qreal DRP) = 0;
    /// area to render content
    virtual QRect renderRect() const = 0;
    /// set area to render content
    virtual void setRenderRect(QRect renderRect) = 0;
    /// current scroll offset along x axis
    virtual double scrollOffsetX() const = 0;
    /// current scroll offset along y axis
    virtual double scrollOffsetY() const = 0;
    /// device form-factor
    virtual int formFactor() const = 0;
    /// set device form-factor
    virtual void setFormFactor(int mode) = 0;
    /// User-Agent override mode
    virtual bool userAgentOverride() const = 0;
    /// sets insets values.
    ///
    /// The values define a rectangle by its top, right, bottom, and left insets
    /// from the edge of the viewport, which is safe to put content into without
    /// risking it being cut off by the shape of a non‑rectangular display. The
    /// CSS global variables (safe-area-inset-*) take that values respectively.
    /// The values take effect only in the fullscreen mode. To reset insets,
    /// pass zeroed inset parameters.
    virtual void setInsets(int top, int right, int bottom, int left) = 0;

    /// load raw HTML text
    virtual void loadHtml(const QString &html) = 0;
    /// clear text selection
    virtual void clearTextSelection() = 0;
    /// run JavaScript snippet in the context of the loaded document
    ///
    /// @link runjs.qml Usage Example @endlink
    virtual void runJavaScript(
        const QString &script, const QJSValue &callback, const QJSValue &errorCallback)
        = 0;
    /// send an asynchronous message with the specified name and data
    ///
    /// @link asyncmessages.qml Usage Example @endlink
    virtual void sendAsyncMessage(const QString &name, const QVariant &data) = 0;
    /// register a listener for asynchronous messages with the specified name
    virtual void addMessageListener(const QString &name) = 0;
    /// exit from fullscreen mode
    virtual void exitFullscreenMode() = 0;
    /// @brief get a PWA manifest URL of current website (if present)
    /// @return manifest URL (empty if does not exist)
    virtual QString pwaManifest() = 0;
    /// indicates if current WebView is a Popup
    virtual bool isPopup() const = 0;
    /// If |isFavicon| is true, cookies are not
    /// sent and not accepted during download
    virtual void downloadImage(const QString &url, bool isFavicon = true) = 0;
    /// download a string resource (e.g., a json file)
    /// content size must not exceed 5mb, otherwise it won't be downloaded
    virtual void downloadStringResource(const QString &url) = 0;

    virtual IRendering *renderingExtension() const = 0;
    virtual IInputExtension *inputExtension() const = 0;
    virtual ITextExtension *textExtension() const = 0;
    virtual IContextMenuExtension *contextMenuExtension() const = 0;
    virtual IPdfPrint *pdfPrintExtension() const = 0;
    virtual IMediaPlaybackExtension *mediaPlaybackExtension() const = 0;
    virtual IOverscrollExtension *overscrollExtension() const = 0;
    virtual ILoadRequestExtension *loadRequestExtension() const = 0;
    virtual IPopupExtension *popupExtension() const = 0;
    virtual IFaviconExtension *faviconExtension() const = 0;
    virtual IConnectionInfoExtension *connectionInfoExtension() const = 0;
    virtual IJavaScriptExtension *javaScriptExtension() const = 0;
    virtual IPasswordManagerExtension *passwordManagerExtension() const = 0;
};

} // namespace WebView
} // namespace Aurora
