/* This file is part of the Aurora SDK
*/

var shortcutName = "Aurora IDE";

function setupBashPathWidget()
{
    installer.addWizardPage(component, "BashPathWidget", QInstaller.ComponentSelection);

    var bashPathWidget = component.userInterface("BashPathWidget");
    if (!bashPathWidget)
        return;

    var pathLineEdit = bashPathWidget.findChild("pathLineEdit");
    var errorLabel = bashPathWidget.findChild("errorLabel");
    var browseButton = bashPathWidget.findChild("browseButton");
    if (!pathLineEdit || !errorLabel || !browseButton)
        return;

    pathLineEdit.text = installer.value("bashPath");
    errorLabel.text = "";
    installer.setValidatorForCustomPage(component, "BashPathWidget", "validateBashPath");
    browseButton.clicked.connect(function() {
        var path = QFileDialog.getOpenFileName(qsTranslate("installscript", "Select Bash Executable"),
                                               pathLineEdit.text, qsTranslate("installscript", "Applications (*.exe)"));
        if (path !== "")
            pathLineEdit.text = path;
    });
}

function Component()
{
    console.log("Component constructor: " + component.name);
    component.setValue("DisplayName", qsTranslate("installscript", "Aurora IDE"));
    component.setValue("Description", qsTranslate("installscript", "The IDE for Aurora OS development."));

    installer.valueChanged.connect( this, Component.prototype.reactOnTargetDirChange );
    // set the default values to SDKToolBinary and QtCreatorInstallerSettingsFile
    Component.prototype.reactOnTargetDirChange("TargetDir", installer.value("TargetDir"));

    var os = installer.value("os");
    var pathSeparator = os === "win" ? ";" : ":";
    var paths = installer.environmentVariable("PATH").split(pathSeparator)

    if (installer.value("os") === "win"
            && !installer.value("bashPath")
            && (installer.isInstaller() || installer.isUpdater())) {
        var defaultLocations = [
            "C:/msys64/usr/bin",
            installer.value("ApplicationsDir") + "/Git/usr/bin",
            installer.value("HomeDir") + "/AppData/Local/Programs/Git/usr/bin"
        ];
	    defaultLocations = defaultLocations.concat(paths);

        var bashPath = "";
        for (var i = 0; i < defaultLocations.length; ++i) {
            bashPath = installer.findPath("bash.exe", defaultLocations[i]);
            console.log("Search bashPath: " + bashPath + " in: " + defaultLocations[i]);
            if (bashPath == "")
                continue;
            bashPath = bashPath.replace(/\\/g, "/");
            var bashDir = bashPath.replace(/[^\/]+$/, "");
            if (installer.fileExists(bashDir + "ssh-keygen.exe")) {
                break;
            }          
         }
    
        installer.setValue("bashPath", bashPath);

        setupBashPathWidget();
        installer.currentPageChanged.connect(this, Component.prototype.handlePageChanged);
    }

    // During the installation SSH keys for the build engine and emulator(s) need to be generated.
    // Note that ssh-keygen is not used directly but through `merssh generatesshkeys`.
    //
    // On Windows it is assumed that SSH tools are collocated with bash and they are not required
    // to be on PATH.
    if (installer.value("os") !== "win") {
        var sshKeygenPath = installer.findPath("ssh-keygen", paths);
        if (!sshKeygenPath) {
            if (!installer.value("abortInstaller")) {
                installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                    qsTranslate("installscript", "OpenSSH client tools not installed: Please ensure OpenSSH client tools " +
                        "are installed and in PATH, then restart the installer"));
                installer.setValue("abortInstaller", true);
                return;
            }
        }
    }
}

Component.prototype.beginInstallation = function()
{
    var os = installer.value('os');

    if (os === "win")
       component.setStopProcessForUpdateRequest("qtcreator.exe", true);
    else if (os === "mac")
	component.setStopProcessForUpdateRequest("Qt Creator", true);
    else
	component.setStopProcessForUpdateRequest("qtcreator", true);
}

Component.prototype.reactOnTargetDirChange = function(key, value)
{
    if (key === "TargetDir") {
        var os = installer.value('os');
        if (os === 'mac') {
            var resources = "/Qt\ Creator.app/Contents/Resources/";
            installer.setValue("QtCreatorInstallerSettingsFile",
                               value + resources + "@ConfigVariant@/QtCreator.ini");
            installer.setValue("LibSfdkInstallerSettingsFile",
                               value + resources + "@ConfigVariant@/libsfdk.ini");
            installer.setValue("SfdkInstallerSettingsFile",
                               value + resources + "@ConfigVariant@/sfdk.ini");
            installer.setValue("SDKToolBinary", value + resources + "libexec/sdktool");
            installer.setValue("merSshPath", value + resources + "libexec/merssh");
            installer.setValue("sfdkPath", value + "/Qt\ Creator.app/Contents/MacOS/sfdk");
        } else {
            var settingsFolder = value + "/share/qtcreator/@ConfigVariant@";
            installer.setValue("QtCreatorInstallerSettingsFile", settingsFolder + "/QtCreator.ini");
            installer.setValue("LibSfdkInstallerSettingsFile", settingsFolder + "/libsfdk.ini");
            installer.setValue("SfdkInstallerSettingsFile", settingsFolder + "/sfdk.ini");
            installer.setValue("SDKToolBinary", (os === 'win')
                                                ? value + "/bin/sdktool"
                                                : value + "/libexec/qtcreator/sdktool");
            installer.setValue("merSshPath", (os === 'win') ?
                                   (value + "\\bin\\merssh.exe").replace(/\//, "\\")
                                 : value + "/libexec/qtcreator/merssh");
            installer.setValue("sfdkPath", value + "/bin/sfdk");
        }
    }
}

Component.prototype.createOperations = function()
{
    installer.installationFinished.connect( this, Component.prototype.installationFinished);
    installer.finishButtonClicked.connect(this, Component.prototype.installationFinishClicked);

    // Call the base createOperations
    component.createOperations();

    var workspaceDir = installer.value("workspaceDir");

    var os = installer.value('os');

    if (os === 'win') {
        var bashDir = installer.value("bashPath").replace(/[^/]+$/, "");
        component.addOperation("Settings",
                               "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                               "method=set",
                               "key=Git/Path",
                               "value=" + bashDir);

        // MSYS2 does not use the real (Windows) user's home directory by default
        var result = installer.execute("@bashPath@", ["-c", "/usr/bin/cygpath --mixed $HOME"]);
        if (result.length < 2 || result[1] !== 0) {
            console.log("Failed to determine Bash home. Result: " + result);
            installer.setValue("bashHome", installer.value("HomeDir"));
        } else {
            var bashHome = result[0].trim();
            console.log("Bash home: '" + bashHome + "'");
            installer.setValue("bashHome", bashHome);
        }
    }

    if (os === 'mac') {
        // sdk-init (executed under build engine) needs equal paths on all platforms
        component.addOperation("Mkdir", "@TargetDir@/share/");
        // We need a relative link, so that it can be resolved under the build engine.
        // The CreateLink operation cannot do that.
        component.addOperation("Execute",
                               "ln", "-s", "../Qt Creator.app/Contents/Resources",
                               "@TargetDir@/share/qtcreator",
                               "UNDOEXECUTE",
                               "rm", "@TargetDir@/share/qtcreator");
    }

    let sdkRelease = installer.value("ConfigSDKRelease");
    if (os === 'x11') {
        var comment = qsTranslate("installscript", "The IDE of choice for Aurora OS development.");
        component.addOperation( "InstallIcons", "@TargetDir@/share/icons", "@ConfigVariant@" );
        component.addOperation( "CreateDesktopEntry",
                               "@ConfigVariant@-qtcreator.desktop",
                               "Type=Application\n" +
                               "Exec=@TargetDir@/bin/qtcreator.sh -name @ConfigVariant@-qtcreator\n" +
                               "Path=" + workspaceDir + "\n" +
                               `Name=${shortcutName} ${sdkRelease}\n` +
                               "GenericName=" + comment + "\n" +
                               "Comment=" + comment + "\n" +
                               "Icon=@ConfigVariant@-qtcreator\n" +
                               "Terminal=false\n" +
                               "StartupWMClass=@ConfigVariant@-qtcreator\n" +
                               "Categories=Development;IDE;Qt;\n" +
                               "MimeType=text/x-c++src;text/x-c++hdr;text/x-xsrc;application/x-designer;text/x-qml;"
                               );
        xdgtool = installer.execute("which", new Array("xdg-desktop-menu"))[0];
        xdgtool = xdgtool.replace(/(\r\n|\n|\r)/gm,"");
        if (xdgtool) {
            component.addOperation("Execute", xdgtool, "forceupdate");
        }
    } else if (os === 'mac') {
        component.addOperation("CreateLink",
                               `@HomeDir@/Applications/${shortcutName} ${sdkRelease}.app`,
                               "@TargetDir@/Qt Creator.app"
                               );
    } else if (os === 'win') {
        component.addOperation("CreateShortcut",
                               "@TargetDir@/bin/qtcreator.exe",
                               `@StartMenuDir@/${shortcutName} ${sdkRelease}.lnk`
                               );
    }

    if (os === 'win') {
        // CreateLink can only create junctions for directories and there is no CreateFile
        component.addOperation("Mkdir",
                               "@bashHome@/.local/share/bash-completion/completions/");
        component.addOperation("AppendFile",
                               "@bashHome@/.local/share/bash-completion/completions/sfdk",
                               '. "$(cygpath --unix \'@TargetDir@/share/qtcreator/sfdk/sfdk.bash-completion\')"');
    } else {
        component.addOperation("CreateLink",
                               "@HomeDir@/.local/share/bash-completion/completions/sfdk",
                               "@TargetDir@/share/qtcreator/sfdk/sfdk.bash-completion");
    }

    // Set default project paths
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=Directories/Projects",
                           "value=" + workspaceDir);
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=Directories/UseProjectsDirectory",
                           "value=true");

    // Disable some unused plugins by default
    [ "Qnx", "Designer", "QmlDesigner", "WinRt", "Marketplace" ].forEach(function(pluginName) {
        component.addOperation("Settings",
                               "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                               "method=add_array_value",
                               "key=Plugins/Ignored",
                               "value=" + pluginName);
    })

    // Settings to make the Updaterplugin in Creator work
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=Updater/MaintenanceTool",
                           "value=" + installer.value("MaintenanceToolPath"));
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=Updater/AutomaticCheck",
                           "value=false");
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=Updater/CheckForNewQtVersions",
                           "value=false");


    // Force some plugins to be enabled
    [ "UpdateInfo", "CompilationDatabaseProjectManager" ].forEach(function(pluginName) {
        component.addOperation("Settings",
                               "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                               "method=add_array_value",
                               "key=Plugins/ForceEnabled",
                               "value=" + pluginName);
    });

    // Ensure that Unix line terminators are used everywhere
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=General/DefaultLineTerminator",
                           "value=0");

    // Avoid use of git-add with '--intent-to-add' - it breaks git-stash
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=Git/AddImmediately",
                           "value=true");

    if (installer.value("buildEngineType") !== "docker") {
        component.addOperation("Settings",
                               "path=" + installer.value("LibSfdkInstallerSettingsFile"),
                               "method=add_array_value",
                               "key=MaskedVmTypes",
                               "value=Docker");
    }
    
    // Turn off Package manager autosetup due to incompatibility
    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=set",
                           "key=CMakeSpecificSettings/PackageManagerAutoSetup",
                           "value=false");
}

Component.prototype.handlePageChanged = function(page)
{
    if (page == QInstaller.ComponentSelection) {
        var bashPathWidget = component.userInterface("BashPathWidget");
        var pathLineEdit = bashPathWidget.findChild("pathLineEdit");
        var bashPath = pathLineEdit.text;
        bashPath = bashPath.replace(/\\/g, "/");

        console.log("Bash path: '" + bashPath + "'");
        installer.setValue("bashPath", bashPath);
    }
}

Component.prototype.validateBashPath = function()
{
    var errorTemplate = '<html><head/><body><p><span style="color:#ef2929;">%1</span></p></body></html>';
    var bashPathWidget = component.userInterface("BashPathWidget");
    var pathLineEdit = bashPathWidget.findChild("pathLineEdit");
    var bashPath = pathLineEdit.text;
    var errorLabel = bashPathWidget.findChild("errorLabel");

    if (bashPath === '') {
        errorLabel.text = errorTemplate.arg(qsTranslate("installscript", "Path may not be empty"));
        return false;
    }

    if (!installer.fileExists(bashPath)) {
        errorLabel.text = errorTemplate.arg(qsTranslate("installscript", "File does not exist"));
        return false;
    }

    bashPath = bashPath.replace(/\\/g, "/");
    var bashDir = bashPath.replace(/[^\/]+$/, "");

    if (!installer.fileExists(bashDir + "ssh-keygen.exe")) {
        errorLabel.text = errorTemplate.arg(qsTranslate("installscript", "The required executable " +
            "\"ssh-keygen.exe\" was not found in this bash installation"));
        return false;
    }

    errorLabel.text = "";
    return true;
}

function updateFinishedText()
{
    var os = installer.value("os");
    var targetDir = installer.value("TargetDir");
    var finishedText;
    if (os === "mac") {
        finishedText = qsTranslate("installscript", "You can launch the IDE from the Launchpad: <em>%1</em>\n" +
                       "or in the Finder: <em>%2</em>").arg(shortcutName).arg(targetDir + "/Qt Creator.app");
    } else if (os === "win") {
        var startMenuDir = installer.value("StartMenuDir").split("\\").pop();
        finishedText = qsTranslate("installscript", "You can launch the IDE from the Start Menu: <em>%1</em>").arg(startMenuDir + "\\" + shortcutName);
    } else {
        finishedText = qsTranslate("installscript", "You can launch the IDE from the command line: <em>%1</em>\n" +
                       "or search for <em>%2</em> in the application manager.").arg(targetDir + "/bin/qtcreator.sh").arg(shortcutName);
    }
    installer.setValue("FinishedText", finishedText);
}

Component.prototype.installationFinished = function()
{
    installer.setValue("FinishedText", "");
    var os = installer.value("os");

    if (installer.isInstaller()) {
        var qtcreatorApp = (installer.value('os') === 'mac') ? "/Qt\ Creator.app/Contents/MacOS/Qt\ Creator"
                                                             : "/bin/qtcreator";
        installer.setValue("RunProgram", installer.value("TargetDir") + qtcreatorApp);
        installer.setValue("RunProgramDescription", qsTranslate("installscript", "Launch Aurora IDE"));
        updateFinishedText();
        return;
    }

    if (installer.isUpdater() && installer.status == QInstaller.Success) {
        finishedText = qsTranslate("installscript", "Please restart Qt Creator to use the latest greatest version.")
        if(os != "mac") finishedText +="\n\nClick Finish to exit the Wizard.";
        else finishedText +="\n\nClick Done to exit the Wizard";
        installer.setValue("FinishedText", finishedText);
        return;
    }

}

Component.prototype.installationFinishClicked = function()
{
    installer.setValue("FinishedText", "");
}
