/* SPDX-FileCopyrightText: 2016-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * SDJD is a system daemon for reliable logging of security related events
 * to preallocated circular buffer on disk.
 */

#ifndef _LIBOMPLOG_H_
#define _LIBOMPLOG_H_

#include <stddef.h> /* For size_t argument in the deprecated send_log() */
#include <stdbool.h>
#include <stdint.h>

/* Make enums more C++-friendly */
#ifdef __cplusplus
#ifndef CLASS
#define CLASS class
#endif
#else
#ifndef CLASS
#define CLASS
#endif
#endif

#ifndef UNUSED
#define UNUSED __attribute__((unused))
#endif

#define SDJD_D_BUS_BUS_TYPE_SYSTEM  (0)
#define SDJD_D_BUS_BUS_TYPE_SESSION (1)
#define SDJD_D_BUS_BUS_TYPE         SDJD_D_BUS_BUS_TYPE_SYSTEM
#define SDJD_D_BUS_BUS_NAME         "ru.omprussia.SystemLogs"
#define SDJD_D_BUS_OBJ_PATH         "/ru/omprussia/SystemLogs"
#define SDJD_D_BUS_INTERFACE_NAME   "ru.omprussia.SystemLogs"
#define SDJD_D_BUS_API_VERSION      "2.1"

/* TODO: It is used by securityd. Refactor it to use some config-file instead
 * of having build-time dependency. */
#define CONFIG_FILE_PATH            "/etc/omp/sdjd.conf"

#define EVENT_ID(ID_, OPTION_, ROLE_, IN_USE_) ID_

/* TODO: Fix the login service.
 * And think about whether to use strings or decimal values (%d) instead? */
#define FOR_EACH_LOGIN_FAIL(E)                                   \
        E(LOGIN_CONFIG_SYNTAX_FAIL, "login-config-syntax-fail"), \
        E(LOGIN_TTL_FAIL,           "login-ttl-fail"),           \
        E(LOGIN_SCHEDULE_FAIL,      "login-schedule-fail"),      \
        E(LOGIN_PASS_FAIL,          "login-pass-fail")

#define SDJD_ROLE_NAME_PRIMARY   "role-primary-user"
#define SDJD_ROLE_NAME_SECONDARY "role-secondary-user"
#define SDJD_ROLE_NAME_SYSTEM    "role-system"
#define SDJD_ROLE_NAME_UNKNOWN   "role-unknown"
#define SDJD_ROLE_NAME_CURRENT   NULL

/* TODO: Drop unused events */
#define FOR_EACH_EVENT(E)                                                                           \
        /* Event identifier,          option string,                overriding role,        in-use? \
         */                                                                                         \
        E(LOGIN_SUCCESSFUL,           "login-successful",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(DEVLOCK_ON,                 "devlock-on",                 SDJD_ROLE_NAME_CURRENT, true),  \
        E(DEVLOCK_ATTEMPTS_EXPIRED,   "devlock-attempts-expired",   SDJD_ROLE_NAME_CURRENT, true),  \
        E(LOGIN_FAILED,               "login-failed",               SDJD_ROLE_NAME_CURRENT, true),  \
        E(OUTDATED_PASS,              "outdated-pass",              SDJD_ROLE_NAME_CURRENT, true),  \
        E(PASS_CHANGE,                "pass-change",                SDJD_ROLE_NAME_CURRENT, true),  \
        E(INTEGRITY_CHECK_START,      "integrity-check-start",      SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(INTEGRITY_CHECK_SUCCESSFUL, "integrity-check-successful", SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(INTEGRITY_FAILED,           "integrity-failed",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(OUTDATING_PASS,             "outdating-pass",             SDJD_ROLE_NAME_CURRENT, true),  \
        E(DEVICE_BLOCKED,             "device-blocked",             SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(INTEGRITY_INIT,             "integrity-init",             SDJD_ROLE_NAME_SYSTEM,  false), \
        E(NEW_PACKAGE_INSTALLED,      "new-package-installed",      SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PACKAGE_UPGRADED,           "package-upgraded",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PACKAGE_REMOVED,            "package-removed",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PACKAGE_FAILED,             "package-failed",             SDJD_ROLE_NAME_SYSTEM,  false), \
        E(ADDED_USB_STORAGE_DEVICE,   "added-usb-storage-device",   SDJD_ROLE_NAME_CURRENT, true),  \
        E(REMOVED_USB_STORAGE_DEVICE, "removed-usb-storage-device", SDJD_ROLE_NAME_CURRENT, true),  \
        E(ADDED_MMC_STORAGE_DEVICE,   "added-mmc-storage-device",   SDJD_ROLE_NAME_CURRENT, true),  \
        E(REMOVED_MMC_STORAGE_DEVICE, "removed-mmc-storage-device", SDJD_ROLE_NAME_CURRENT, true),  \
        E(ADDED_TOKEN_DEVICE,         "added-token-device",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(REMOVED_TOKEN_DEVICE,       "removed-token-device",       SDJD_ROLE_NAME_CURRENT, true),  \
        E(END_OF_LOG_FILE,            "end-of-log-file",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(REBOOTED,                   "rebooted",                   SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SHUTDOWN,                   "shutdown",                   SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(STARTED_UP,                 "started-up",                 SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(CRASHED,                    "crashed",                    SDJD_ROLE_NAME_SYSTEM,  false), \
        E(RESTARTED_INCORRECTLY,      "restarted-incorrectly",      SDJD_ROLE_NAME_CURRENT, true),  \
        E(IPTABLES_RULES_ADDED,       "iptables-rules-added",       SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(IPTABLES_RULES_FAILURE,     "iptables-rules-failure",     SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SDJD_STARTED,               "sdjd-started",               SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SDJD_STOPPED,               "sdjd-stopped",               SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(AUDIT_CFG_CHANGED,          "audit-cfg-changed",          SDJD_ROLE_NAME_CURRENT, true),  \
        /* TODO: Ensure that this event is unused event. Check it. */                               \
        E(SEND_LOG_USER_ID,           "send-log-user-id",           SDJD_ROLE_NAME_CURRENT, false), \
        E(TIME_CHANGED,               "time-changed",               SDJD_ROLE_NAME_CURRENT, true),  \
        E(APP_CRASHED,                "app-crashed",                SDJD_ROLE_NAME_CURRENT, true),  \
        /* TODO: Drop third-party events */                                                         \
        E(THIRD_PARTY_INFO,           "third-party-info",           SDJD_ROLE_NAME_CURRENT, false), \
        E(THIRD_PARTY_WARN,           "third-party-warn",           SDJD_ROLE_NAME_CURRENT, false), \
        E(THIRD_PARTY_DEBUG,          "third-party-debug",          SDJD_ROLE_NAME_CURRENT, false), \
        E(THIRD_PARTY_ALERT,          "third-party-alert",          SDJD_ROLE_NAME_CURRENT, false), \
        E(AUDIT_EACCES,               "audit-eacces",               SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PACKAGE_FAILED_VALIDATION,  "package-failed-validation",  SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PACKAGE_FAILED_OMPCERT,     "package-failed-ompcert",     SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PACKAGE_FAILED_RPM,         "package-failed-rpm",         SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(USER_CREATE,                "user-create",                SDJD_ROLE_NAME_CURRENT, true),  \
        E(USER_CHANGE,                "user-change",                SDJD_ROLE_NAME_CURRENT, true),  \
        E(USER_REMOVE,                "user-remove",                SDJD_ROLE_NAME_CURRENT, true),  \
        E(AUTH_2FA_INIT,              "auth-2fa-init",              SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(AUTH_2FA_SUCCESS,           "auth-2fa-success",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(AUTH_2FA_FAILED,            "auth-2fa-failed",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(CHECKLIST_SUCCESS,          "checklist-success",          SDJD_ROLE_NAME_CURRENT, true),  \
        E(CHECKLIST_FAILED,           "checklist-failed",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_SERVICE_STARTED,         "av-service-started",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_SERVICE_STOPPED,         "av-service-stopped",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_APPLICATION_STARTED,     "av-application-started",     SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_APPLICATION_STOPPED,     "av-application-stopped",     SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_SCAN_STARTED,            "av-scan-started",            SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_SCAN_FINISHED,           "av-scan-finished",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_UPDATE_FINISHED,         "av-update-finished",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_AUDIT_STARTED,           "av-audit-started",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_AUDIT_STOPPED,           "av-audit-stopped",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_THREAT_DETECTED,         "av-threat-detected",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_INFO,                    "av-info",                    SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_ERROR,                   "av-error",                   SDJD_ROLE_NAME_CURRENT, true),  \
        E(AV_SCAN_POSTPONED,          "av-scan-postponed",          SDJD_ROLE_NAME_CURRENT, true),  \
        E(TEE_STARTED,                "tee-started",                SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_STORE_LOCKED,           "tee-store-locked",           SDJD_ROLE_NAME_SYSTEM,  false), \
        E(TEE_STORE_UNLOCKED,         "tee-store-unlocked",         SDJD_ROLE_NAME_SYSTEM,  false), \
        E(TEE_NORMAL_BOOT,            "tee-normal-boot",            SDJD_ROLE_NAME_SYSTEM,  false), \
        E(TEE_INTRUSION_DETECTED,     "tee-intrusion-detected",     SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_PASSWORD_SKIPPED,       "tee-password-skipped",       SDJD_ROLE_NAME_SYSTEM,  false), \
        E(TEE_PASSWORD_CHANGED,       "tee-password-changed",       SDJD_ROLE_NAME_SYSTEM,  false), \
        E(TEE_PASSWORD_RESET,         "tee-password-reset",         SDJD_ROLE_NAME_SYSTEM,  false), \
        E(TEE_PASSWORD_OK,            "tee-password-ok",            SDJD_ROLE_NAME_SYSTEM,  false), \
        E(LOG_VIEWER_STARTED,         "log-viewer-started",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(INTEGRITY_3RDPARTY_FAILED,  "integrity-3rdparty-failed",  SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PASSWORD_POLICY_CHANGED,    "password-policy-changed",    SDJD_ROLE_NAME_CURRENT, true),  \
        E(USER_POLICY_CHANGED,        "user-policy-changed",        SDJD_ROLE_NAME_CURRENT, true),  \
        E(DEVELOPER_MODE_ENABLED,     "developer-mode-enabled",     SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(REMOTE_LOGIN_ENABLED,       "remote-login-enabled",       SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(REMOTE_LOGIN_DISABLED,      "remote-login-disabled",      SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SUPERUSER_PASSWORD_CHANGED, "superuser-password-changed", SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TERMINAL_ENABLED,           "terminal-enabled",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(TERMINAL_DISABLED,          "terminal-disabled",          SDJD_ROLE_NAME_CURRENT, true),  \
        E(SD_DRIVE_MOUNTED,           "sd-drive-mounted",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(SD_DRIVE_UNMOUNTED,         "sd-drive-unmounted",         SDJD_ROLE_NAME_CURRENT, true),  \
        E(USB_DRIVE_MOUNTED,          "usb-drive-mounted",          SDJD_ROLE_NAME_CURRENT, true),  \
        E(USB_DRIVE_UNMOUNTED,        "usb-drive-unmounted",        SDJD_ROLE_NAME_CURRENT, true),  \
        E(MTP_MODE_ON,                "mtp-mode-on",                SDJD_ROLE_NAME_CURRENT, true),  \
        E(MTP_MODE_OFF,               "mtp-mode-off",               SDJD_ROLE_NAME_CURRENT, true),  \
        E(USB_SSH_MODE_ON,            "usb-ssh-mode-on",            SDJD_ROLE_NAME_CURRENT, true),  \
        E(USB_SSH_MODE_OFF,           "usb-ssh-mode-off",           SDJD_ROLE_NAME_CURRENT, true),  \
        E(AUTH_FACTORS_CHANGED,       "auth-factors-changed",       SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_USING_EXISTED_HUK,      "tee-using-existed-huk",      SDJD_ROLE_NAME_SYSTEM,  false), \
        E(PRINTER_ADDED,              "printer-added",              SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(PRINTER_REMOVED,            "printer-removed",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SMARTCARD_BINDED,           "smartcard-binded",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SMARTCARD_UNBINDED,         "smartcard-unbinded",         SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SMARTCARD_ERROR,            "smartcard-error",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(SMARTCARD_WARNING,          "smartcard-warning",          SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(AUTH_FAILED,                "auth-failed",                SDJD_ROLE_NAME_CURRENT, true),  \
        E(AUTH_SUCCESSFUL,            "auth-successful",            SDJD_ROLE_NAME_CURRENT, true),  \
        E(SECURITY_POLICY_CHANGED,    "security-policy-changed",    SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(AUTH_OVER_HTTP_ALLOWED,     "auth-over-http-allowed",     SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(AUTH_OVER_HTTP_DENIED,      "auth-over-http-denied",      SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(APP_ATTESTATION_OK,         "app-attestation-ok",         SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(APP_ATTESTATION_FAILED,     "app-attestation-failed",     SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(DEVICE_ATTESTATION_OK,      "device-attestation-ok",      SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(DEVICE_ATTESTATION_FAILED,  "device-attestation-failed",  SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_BOOT_NORMAL,            "tee-boot-normal",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_BOOT_ERROR,             "tee-boot-error",             SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_CRYPTO_ERROR,           "tee-crypto-error",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_GENERIC_INFO,           "tee-generic-info",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_GENERIC_WARN,           "tee-generic-warn",           SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_GENERIC_ALERT,          "tee-generic-alert",          SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_TA_SIGN_NOT_VERIFIED,   "tee-ta-sign-not-verified",   SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_TA_GENERIC_INFO,        "tee-ta-generic-info",        SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_TA_GENERIC_WARN,        "tee-ta-generic-warn",        SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(TEE_TA_GENERIC_ALERT,       "tee-ta-generic-alert",       SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(USB_WHITELIST_ACTIVATED,    "usb-whitelist-activated",    SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(USB_WHITELIST_DEACTIVATED,  "usb-whitelist-deactivated",  SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(USB_WHITELIST_CHANGED,      "usb-whitelist-changed",      SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(RESET_ACTIVATED,            "reset-activated",            SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(RESET_FINISHED,             "reset-finished",             SDJD_ROLE_NAME_SYSTEM,  true),  \
        E(RESET_FAILED,               "reset-failed",               SDJD_ROLE_NAME_SYSTEM,  true)

        /* Note: After adding the new event here don't forget to update
         * the configuration file.
         * When you are removing an event, never delete the line, but simply
         * change the "in-use" field to "false" instead. */

#define LOGIN_FAIL_ID(ID_, REASON_) ID_

enum CLASS sdjd_login_fail {
	FOR_EACH_LOGIN_FAIL(LOGIN_FAIL_ID)
};

enum CLASS sdjd_event {
	FOR_EACH_EVENT(EVENT_ID),
	NUMBER_OF_EVENTS /* Not a real event */
};

/* TODO: Change the meaning of the level and fix numeration. */
#define	INVALID_LEVEL   (0)      /* not log */
#define	INFO_LEVEL      (1)      /* log only into bin log  */
#define	WARN_LEVEL      (1 << 1) /* log in bin and journal */
#define	DEBUG_LEVEL     (1 << 2) /* log into stderr */
#define	ALERT_LEVEL     (1 << 3) /* log into bin, journal, dbus, etc... */

/* TODO: Is it used by other apps? */
#define IS_INVALID(lev) (0 == (lev))
#define IS_INFO(lev)    ((lev) & INFO_LEVEL)
#define IS_WARN(lev)    ((lev) & WARN_LEVEL)
#define IS_DEBUG(lev)   ((lev) & DEBUG_LEVEL)
#define IS_ALERT(lev)   ((lev) & ALERT_LEVEL)

/* TODO: What is it for? */
#define LOG_USER_ID     33

/* Return value of send_log() and log_event() */
#define SEND_LOG_SUCCESS         0
/* Not used anymore,
 * for any bad parameters SEND_LOG_BAD_PARAMS will be returned */
#define SEND_LOG_UNKNOWN_ID     -1
#define SEND_LOG_BAD_PARAMS     -2
#define SEND_LOG_INTERNAL_ERROR -3

/* Return value of apply_filter() */
#define APPLY_FILTER_SUCCESS         0
#define APPLY_FILTER_REPEAT         -1
#define APPLY_FILTER_BAD_PARAMS     -2
#define APPLY_FILTER_INTERNAL_ERROR -3

#ifdef  __cplusplus
extern "C" {
#endif

#define API_EXPORT __attribute__ ((visibility ("default")))

/* Maximal length of the message. */
#define SDJD_MESSAGE_LEN_MAX 8192

/* This function is deprecated */
int send_log(uint32_t level, int cmd_id, const void *data, size_t len)
    __attribute__((deprecated("Use log_event() instead"))) API_EXPORT;

/* Sends an event to sdjd synchronously via D-Bus in blocking mode.
 * level:   Importance level
 * type:    Event type
 * message: 0-terminated optional message of up to 4096 bytes */
int log_event(unsigned char level, enum sdjd_event type, const char *message)
    API_EXPORT;

#ifdef __cplusplus
}
#endif

#endif /* _LIBOMPLOG_H_ */
