/*
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>
#include <QObject>

class QQmlEngine;
class QJSEngine;

namespace Aurora {
namespace WebView {

class BrowsingDataManagerPrivate;

/**
 * @class BrowsingDataManager
 * @brief can be used to manage website data.
 * 
 * BrowsingDataManager is a global singleton class.
 * Use it only when isReady() is true, otherwise
 * setters/getters are not guaranteed to work as intended.
 */
class BrowsingDataManager : public QObject
{
    Q_OBJECT
    /// BrowsingDataManager can be used only when `ready` is true
    Q_PROPERTY(bool ready READ isReady NOTIFY readyChanged)
public:
    /**
     * @brief Default destructor
     */
    virtual ~BrowsingDataManager();
    /**
     * @brief returns instance of a global BrowsingDataManager
     */
    static QSharedPointer<BrowsingDataManager> GetInstance(
        QQmlEngine *qmlEngine = nullptr, QJSEngine *jsEngine = nullptr);

    /**
     * @brief retrieves browsing data info for a given url.
     * Result comes with siteDataInfoAvailable() signal.
     * 
     * To ensure that info is not outdated, firstly call updateBrowsingDataInfo().
     * 
     * @return true if url is correct and browsing data manager can be accessed.
     */
    Q_INVOKABLE bool getSiteDataInfo(const QString &url);
    /**
     * @brief retrieves info about overall browsing data size.
     * Result comes with totalBrowsingDataInfoAvailable() signal.
     * 
     * To ensure that info is not outdated, firstly call updateBrowsingDataInfo().
     * 
     * @return true if browsing data manager can be accessed.
     */
    Q_INVOKABLE bool getTotalBrowsingDataSize();

    /**
     * @brief deletes browsing data of a given url.
     * Result comes with siteDataCleared() signal.
     * 
     * @return true if url is correct and browsing data manager can be accessed.
     */
    Q_INVOKABLE bool clearSiteData(const QString &url);
    /**
     * @brief deletes overall browsing data (for all websites).
     * Result comes with overallBrowsingDataCleared() signal.
     * 
     * @param clearSiteData pass true to clear site data
     * @param clearHttpCache pass true to clear HTTP cache as well
     * 
     * @return true if url is correct and browsing data manager can be accessed.
     */
    Q_INVOKABLE bool clearOverallBrowsingData(bool clearSiteData, bool clearHttpCache);

    /**
     * @brief Retrieves the list of top-level sites (based on eTLD+1 concept)
     * for which associated browsing data was detected.
     * Result comes with topLevelSitesAvailable() signal.
     * 
     * To ensure that info is not outdated, firstly call updateBrowsingDataInfo().
     * 
     * @return true if browsing data manager can be accessed.
     */
    Q_INVOKABLE bool getTopLevelSitesList();
    /**
     * Call this method to trigger a rebuild of browsing data model tree (implementation
     * specifics) and related cache recalculations before using any getters. Otherwise
     * the info returned might be outdated.
     * 
     * This operation is expensive, so don't use it irresponsibly.
     */
    Q_INVOKABLE bool updateBrowsingDataInfo();

    /**
     * @brief BrowsingDataManager must only be used when isReady() returns true.
     * 
     * @return true when BrowsingDataManager can be used.
     */
    bool isReady() const;

signals:
    /// @brief BrowsingDataManager can only be used after readyChanged() is emitted
    void readyChanged();
    void siteDataInfoAvailable(
        QString url,
        quint64 allCookiesCount,
        quint64 partitionedCookiesCount,
        quint64 totalStorageSize,
        quint64 partitionedStorageSize);
    void totalBrowsingDataInfoAvailable(
        qint64 totalBytes, qint64 browserSiteDataSize, qint64 browserCacheSize);

    void siteDataCleared();
    void overallBrowsingDataCleared();

    void topLevelSitesAvailable(const QStringList sites);
    void browsingDataInfoUpdated();

private:
    BrowsingDataManager();

    std::unique_ptr<BrowsingDataManagerPrivate> m_d;
};
} // namespace WebView
} // namespace Aurora
