/****************************************************************************
**
** SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/

#ifndef AC_LIB_QT5_HPP
#define AC_LIB_QT5_HPP

#include <QObject>
#include <QScopedPointer>
#include <QString>

#include "client-credentials.hpp"

namespace aurora {

namespace attestation {

class ClientPrivate;
/*!
 * @class Client
 * @brief Provides interface for interaction with attestation service.
 *
 * This class is designed to retrieve information about OS and application
 * attestation status. It includes information about actual attestation status,
 * last device attestation status and last application attestation status.
 */
class Client : public QObject
{
    Q_OBJECT
public:
    Client();
    ~Client();

    // Async methods
    /*! Request actual attestation status for application, device or both.  
        @param[in] nonce unsigned 32-bit integer passed by application to validate report consistency on receive
        @param[in] level level of requested attestation. @see Level.

        @return Will return void, async function, result will delivered with signal @see reportUpdated.
     */
    void requestAttestation(uint32_t nonce, Level level);
    /*! Request last application attestation status.  

        @return Will return void, async function, result will delivered with signal @see reportUpdated.
     */
    void requestLastAttestation();
    /*! Request last device attestation status.  

        @return Will return void, async function, result will delivered with signal @see reportUpdated.
     */
    void requestLastDeviceAttestation();

    /*! Get attestation status after request finish.  Should be checked after @see reportUpdated received.
        It will return status relying on checks requested by user. For last device or application
        (actual or last) corresponding final status will be received. If both device and application
        attestation will be requested, result will be positive only if both device and application successfully
        passed attestation. Otherwise, status is failed.

        @return Will return attestation status. @see Status.
     */
    Status status() const;
    /*! Get nonce from report response. This parameter should be checked by application to compare the nonce,
        passed by application when requesting actual attestation status. This will help to determine, that 
        returned attestation report is the one, requested by application and wasn't requested by someone else.

        @return Will return nonce containing in report.
     */
    uint32_t nonce() const;
    /*! Get raw report returned by attestation service. Contains report in JSON format, allowing application to
        parse it and make additional decisions.

        @return Will return raw attestation report in JSON format.
    */
    QString rawReport() const;

signals:
    /*! Signal that will be emitted if error during report request will happen.
        @param[out] message contains error message
     */
    void error(const QString &message);
    /*! Signal that will be emitted when report will be received.
        @param[out] report contains attestation report
     */
    void reportUpdated(const QString &report);
    /*! Signal that will be emitted when device attestation report will appear.
        Unlike reportUpdated - this signal is emitted when anyone requested device
        attestation, including regular periodic checks. @see reportUpdated.
        @param[out] report contains device attestation report
    */
    void deviceAttestationFinished(const QString &report);

private:
    Client(const Client &) = delete;
    void operator=(const Client &) = delete;

    QScopedPointer<ClientPrivate> m_clientPrivate;
};

} // namespace attestation
} // namespace aurora

#endif // AC_LIB_QT5_HPP
