/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef MARKET_LIB_QUEUE_OPERATION_REQUEST_HPP
#define MARKET_LIB_QUEUE_OPERATION_REQUEST_HPP

#include "globals.hpp"
#include "operation.hpp"

#include <memory>

namespace Market {

class OperationRequestPrivate;

/**
 * @brief Class of request for operation creation.
 * @class OperationRequest operation-request.hpp <market/operation-request.hpp>
 */
class MARKET_LIB_EXPORT_SYMBOL OperationRequest
{
public:
    class BuilderPrivate;

    /**
     * @brief Class for building requests to create an operation
     * @class Builder operation-request.hpp <market/operation-request.hpp>
     */
    class MARKET_LIB_EXPORT_SYMBOL Builder
    {
    public:
        Builder();

        /**
         * @brief Set the type of operation request.
         *
         * @param type Operation type.
         *
         * @return Reference to itself.
         */
        Builder &SetOperationType(OperationType type);

        /**
         * @brief Set the package ID for operation request with @ref PackageRemove type.
         *
         * @param packageID Package ID.
         *
         * @return Reference to itself.
         */
        Builder &SetPackageID(const std::string &packageID);

        /**
         * @brief Set the package path for operation request with @ref PackageInstall
         *        or @ref PackageUpgrade type.
         *
         * @param packagePath Package path.
         *
         * @return Reference to itself.
         */
        Builder &SetPackagePath(const std::string &packagePath);

        /**
         * @brief Build an operation request with validation of all previously set fields.
         *
         * @return Operation request corresponding to the previously set fields.
         */
        OperationRequest Build() const;

    private:
        std::shared_ptr<BuilderPrivate> m_pimpl;
    };

    /**
     * @brief Get the type of operation from the request.
     *
     * @return Operation type.
     */
    OperationType GetOperationType() const;

    /**
     * @brief Get the package ID from the request.
     *
     * @note The package ID can be empty string if the type of operation
     *       in the request is @ref PackageInstall or @ref PackageUpgrade.
     *
     * @return Package ID.
     */
    const std::string &GetPackageID() const;


    /**
     * @brief Get the package path from the request.
     *
     * @note The package path can be empty string if the type of operation
     *       in the request is @ref PackageRemove.
     *
     * @return Package ID.
     */
    const std::string &GetPackagePath() const;

private:
    friend class BuilderPrivate;
    OperationRequest();

    void SetOperationType(OperationType type);
    void SetPackageID(const std::string &packageID);
    void SetPackagePath(const std::string &packagePath);

    std::shared_ptr<OperationRequestPrivate> m_pimpl;
};

} /* namespace Market */

#endif /* MARKET_LIB_QUEUE_OPERATION_REQUEST_HPP */
