/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef COMMON_ERROR_CODE_HPP
#define COMMON_ERROR_CODE_HPP

#ifdef __cplusplus
#define ERROR_CODE_ENUM_ENTRY(CPP, C) CPP
#else
#define ERROR_CODE_ENUM_ENTRY(CPP, C) ERROR_CODE_##C
#endif

#ifdef __cplusplus
/**
 * @brief General data types and enums
 * @namespace Common
 */
namespace Common {
#endif

/**
 * @brief Error codes that the market daemon can return during the operation.
 * @enum ErrorCode
 */
enum ErrorCode {
    /** No error */
    ERROR_CODE_ENUM_ENTRY(None, NONE) = 0,
    /** Any filesystem error */
    ERROR_CODE_ENUM_ENTRY(Filesystem, FILESYSTEM),
    /** Any internal market daemon error */
    ERROR_CODE_ENUM_ENTRY(Internal, INTERNAL),
    /** The path to the installation package does not exist */
    ERROR_CODE_ENUM_ENTRY(FileNotFound, FILE_NOT_FOUND),
    /** Invalid package (package validation failed) */
    ERROR_CODE_ENUM_ENTRY(BadPackage, BAD_PACKAGE),
    /** Invalid package ID */
    ERROR_CODE_ENUM_ENTRY(BadPackageID, BAD_PACKAGE_ID),
    /** The architecture of the package does not match the architecture of the system */
    ERROR_CODE_ENUM_ENTRY(BadPackageArchitecture, BAD_PACKAGE_ARCHITECTURE),
    /** Invalid package signature */
    ERROR_CODE_ENUM_ENTRY(BadPackageSignature, BAD_PACKAGE_SIGNATURE),
    /** Attempt to downgrade the package version */
    ERROR_CODE_ENUM_ENTRY(DowngradeAttempt, DOWNGRADE_ATTEMPT),
    /** Upgrade for the package has already been prepared, the application is expected to close */
    ERROR_CODE_ENUM_ENTRY(UpgradeAlreadyPrepared, UPGRADE_ALREADY_PREPARED),
    /** Package does not exist */
    ERROR_CODE_ENUM_ENTRY(PackageNotExist, PACKAGE_NOT_EXIST),
    /** Attempt to obtain data about a corrupted package */
    ERROR_CODE_ENUM_ENTRY(CorruptedInstalledPackage, CORRUPTED_INSTALLED_PACKAGE),
    /** Failed to install package dependencies */
    ERROR_CODE_ENUM_ENTRY(UnresolvedDependency, UNRESOLVED_DEPENDENCY) __attribute__((deprecated)),
    /** No permissions to perform operation */
    ERROR_CODE_ENUM_ENTRY(UnauthorizedAccess, UNAUTHORIZED_ACCESS),
    /** Attempt to install package with system developer signature */
    ERROR_CODE_ENUM_ENTRY(SystemPackageInstallationAttempt, SYSTEM_PACKAGE_INSTALLATION_ATTEMPT),
    /** Installing packages without a source signature is not allowed by the administrator */
    ERROR_CODE_ENUM_ENTRY(PackagesWithoutSourceNotAllowed, PACKAGES_WITHOUT_SOURCE_NOT_ALLOWED),
    /** The administrator rejected the request to add the package source to the trusted list */
    ERROR_CODE_ENUM_ENTRY(SourceRejectedByAdministrator, SOURCE_REJECTED_BY_ADMINISTRATOR),
    /** The current user does not have rights to add a new trusted source */
    ERROR_CODE_ENUM_ENTRY(UserNotAuthorizedToManageTrustedSources, USER_NOT_AUTHORIZED_TO_TRUST_THE_SOURCE)
};

#ifndef __cplusplus
typedef enum ErrorCode ErrorCode;
#endif

#ifdef __cplusplus
} /* namespace Common */
#endif

#endif /* COMMON_ERROR_CODE_HPP */
