// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef TOOLS_JSON_SCHEMA_COMPILER_UTIL_H_
#define TOOLS_JSON_SCHEMA_COMPILER_UTIL_H_

#include <memory>
#include <string>
#include <utility>
#include <vector>

#include "base/format_macros.h"
#include "base/strings/stringprintf.h"
#include "base/strings/utf_string_conversions.h"
#include "base/values.h"

namespace json_schema_compiler {

namespace util {

// Populates the item |out| from the value |from|. These are used by template
// specializations of |Get(Optional)ArrayFromList|.
bool PopulateItem(const base::Value& from, std::unique_ptr<base::Value>* out);

bool PopulateItem(const base::Value& from, int* out);
bool PopulateItem(const base::Value& from, int* out, std::u16string* error);
bool PopulateItem(const base::Value& from, bool* out);
bool PopulateItem(const base::Value& from, bool* out, std::u16string* error);
bool PopulateItem(const base::Value& from, double* out);
bool PopulateItem(const base::Value& from, double* out, std::u16string* error);
bool PopulateItem(const base::Value& from, std::string* out);
bool PopulateItem(const base::Value& from,
                  std::string* out,
                  std::u16string* error);
bool PopulateItem(const base::Value& from, std::vector<uint8_t>* out);
bool PopulateItem(const base::Value& from,
                  std::vector<uint8_t>* out,
                  std::u16string* error);
bool PopulateItem(const base::Value& from,
                  std::unique_ptr<base::Value>* out,
                  std::u16string* error);
bool PopulateItem(const base::Value& from, std::unique_ptr<base::Value>* out);

// This template is used for types generated by tools/json_schema_compiler.
template <class T>
bool PopulateItem(const base::Value& from, T* out) {
  if (!from.is_dict())
    return false;
  T obj;
  if (!T::Populate(from, &obj))
    return false;
  *out = std::move(obj);
  return true;
}

// This template is used for types generated by tools/json_schema_compiler with
// error generation enabled.
template <class T>
bool PopulateItem(const base::Value& from, T* out, std::u16string* error) {
  T obj;
  if (!T::Populate(from, &obj, error))
    return false;
  *out = std::move(obj);
  return true;
}

// Populates |out| with |list|. Returns false if there is no list at the
// specified key or if the list has anything other than |T|.
template <class T>
bool PopulateArrayFromList(const base::Value::ConstListView& list,
                           std::vector<T>* out) {
  out->clear();
  T item;
  for (const auto& value : list) {
    if (!PopulateItem(value, &item))
      return false;
    // T might not be movable, but in that case it should be copyable, and this
    // will still work.
    out->push_back(std::move(item));
  }

  return true;
}

// Populates |out| with |list|. Returns false and sets |error| if there is no
// list at the specified key or if the list has anything other than |T|.
template <class T>
bool PopulateArrayFromList(const base::Value::ConstListView& list,
                           std::vector<T>* out,
                           std::u16string* error) {
  out->clear();
  T item;
  std::u16string item_error;
  for (size_t i = 0; i < list.size(); ++i) {
    if (!PopulateItem(list[i], &item, &item_error)) {
      DCHECK(error->empty());
      *error = base::ASCIIToUTF16(
          base::StringPrintf("Parsing array failed at index %" PRIuS ": %s", i,
                             base::UTF16ToASCII(item_error).c_str()));
      return false;
    }
    out->push_back(std::move(item));
  }

  return true;
}

// Creates a new vector containing |list| at |out|. Returns
// true on success or if there is nothing at the specified key. Returns false
// if anything other than a list of |T| is at the specified key.
template <class T>
bool PopulateOptionalArrayFromList(const base::Value::ConstListView& list,
                                   std::unique_ptr<std::vector<T>>* out) {
  out->reset(new std::vector<T>());
  if (!PopulateArrayFromList(list, out->get())) {
    out->reset();
    return false;
  }
  return true;
}

template <class T>
bool PopulateOptionalArrayFromList(const base::Value::ConstListView& list,
                                   std::unique_ptr<std::vector<T>>* out,
                                   std::u16string* error) {
  out->reset(new std::vector<T>());
  if (!PopulateArrayFromList(list, out->get(), error)) {
    out->reset();
    return false;
  }
  return true;
}

// Appends a Value newly created from |from| to |out|. These used by template
// specializations of |Set(Optional)ArrayToList|.
void AddItemToList(const int from, base::ListValue* out);
void AddItemToList(const bool from, base::ListValue* out);
void AddItemToList(const double from, base::ListValue* out);
void AddItemToList(const std::string& from, base::ListValue* out);
void AddItemToList(const std::vector<uint8_t>& from, base::ListValue* out);
void AddItemToList(const std::unique_ptr<base::Value>& from,
                   base::ListValue* out);
void AddItemToList(const std::unique_ptr<base::DictionaryValue>& from,
                   base::ListValue* out);

// This template is used for types generated by tools/json_schema_compiler.
template <class T>
void AddItemToList(const std::unique_ptr<T>& from, base::ListValue* out) {
  out->GetList().Append(base::Value::FromUniquePtrValue(from->ToValue()));
}

// This template is used for types generated by tools/json_schema_compiler.
template <class T>
void AddItemToList(const T& from, base::ListValue* out) {
  out->GetList().Append(base::Value::FromUniquePtrValue(from.ToValue()));
}

// Set |out| to the the contents of |from|. Requires PopulateItem to be
// implemented for |T|.
template <class T>
void PopulateListFromArray(const std::vector<T>& from, base::ListValue* out) {
  out->ClearList();
  for (const T& item : from)
    AddItemToList(item, out);
}

// Set |out| to the the contents of |from| if |from| is not null. Requires
// PopulateItem to be implemented for |T|.
template <class T>
void PopulateListFromOptionalArray(const std::unique_ptr<std::vector<T>>& from,
                                   base::ListValue* out) {
  if (from)
    PopulateListFromArray(*from, out);
}

template <class T>
std::unique_ptr<base::Value> CreateValueFromArray(const std::vector<T>& from) {
  std::unique_ptr<base::ListValue> list(new base::ListValue());
  PopulateListFromArray(from, list.get());
  return std::move(list);
}

}  // namespace util
}  // namespace json_schema_compiler

#endif  // TOOLS_JSON_SCHEMA_COMPILER_UTIL_H_
