#ifndef JOB_QUEUE_H
#define JOB_QUEUE_H

#include <atomic>
#include <condition_variable>
#include <deque>
#include <functional>
#include <mutex>

struct job_queue_t
{
    void add(std::function<void()> &&job)
    {
        std::lock_guard<decltype(mutex_)> func_queue_lock(mutex_);
        queue_.emplace_back(job);
        empty_ = false;
        condvar_.notify_one();
    }

    void add(std::function<void()> &&job, const char *file, int line)
    {
        std::lock_guard<decltype(mutex_)> func_queue_lock(mutex_);
        queue_.emplace_back([job, file, line] { job(); });
        empty_ = false;
        condvar_.notify_one();
    }

    bool exec_one()
    {
        if (empty())
            return false;

        std::function<void()> func;
        {
            std::lock_guard<decltype(mutex_)> func_queue_lock(mutex_);
            std::swap(func, queue_.front());
            queue_.pop_front();
            if (queue_.empty())
                empty_ = true;
        }
        func();
        return true;
    }

    size_t size()
    {
        std::lock_guard<decltype(mutex_)> func_queue_lock(mutex_);
        return queue_.size();
    }

    bool empty() { return empty_; }

    void wait_job()
    {
        std::unique_lock<decltype(mutex_)> lock(mutex_);
        condvar_.wait(lock, [this]() { return queue_.size() > 0; });
    }

    void wait_job_cancel()
    {
        add([]() {});
    }

    std::deque<std::function<void()>> queue_;
    std::mutex mutex_;
    std::atomic_bool empty_ = ATOMIC_VAR_INIT(true);
    std::condition_variable condvar_;
};

#endif // JOB_QUEUE_H
