/* This file is part of the Aurora OS SDK
*/

var archiveVersion = "sdk-build-tools-5.2.0.45-aarch64.tar";
var sdkWrappersDir = "@TargetDir@/sdk";
var releaseName = "5.2.0.45";

function toNativePath(path)
{
    if (installer.value("os") === "win")
        return path.replace(/\//g, "\\");
    else
        return path;
}

function getDockerTag(appendUser = true)
{
    // On Windows (newer Docker versions) & macOS
    // Docker containers are located in user home directory
    // on Linux they are system-wide, so we make them independent per user
    var tag = "aurora-build-tools";
    if (appendUser && (installer.value("os") === "x11")) {
        const user = installer.environmentVariable("USERNAME")
            || installer.environmentVariable("USER");
        tag += `-${user}`;
    }
    tag += `:${releaseName}`;
    return tag;
}

function componentExists(componentName)
{
    var retVal = false;
    installer.components().forEach(function (component) {
        if (component.name === componentName)
            retVal = true;
    });
    return retVal;
}

function dockerAvailable()
{
    return installer.value("dockerPath");
}

function dockerBuildxInstalled()
{
  var result = installer.execute(installer.value("dockerPath"), ["build", "--help"])[0];
  return (typeof result === 'undefined')
    ? false
    : result.includes("buildx build");
}

function dockerUsable()
{
    result = installer.execute(installer.value("dockerPath"), "ps");
    if (result.length < 2) {
        console.log("could not execute " + installer.value("dockerPath"));
        return false;
    }
    if (result[1] !== 0)
        console.log("docker returned " + result[1]);
    return result[1] === 0;
}

function Component()
{
    console.log("Component constructor: " + component.name);
    component.setValue("DisplayName", qsTranslate("installscript", "Aurora SDK %1 Build tools").arg(releaseName));
    component.setValue("Description", qsTranslate("installscript", "Enables development for devices running Aurora OS %1").arg(releaseName));
}

Component.prototype.createOperations = function()
{
    // Call the base createOperations
    component.createOperations();

    console.log("Component createOperations: " + component.name);
    
    registerSDKInQtCreator();
}

function registerSDKInQtCreator()
{
    var installDir = installer.value("TargetDir");
    var homeDir = installer.value("homeDir");
    var wrappersDir = installDir + "/sdk/" + releaseName + "/tools";
    
    var workspaceDir = installer.value("workspaceDir");
    
    createWrappers(homeDir, installDir, releaseName, wrappersDir);
    createKits(installDir, releaseName, wrappersDir);

    component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=add_array_value",
                           "key=SDKBuildTools/Containers",
                           "value=" + getDockerTag() + "|" + wrappersDir)
    component.addOperation("Settings",
                            "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                            "method=set",
                            "key=Mer/WorkspaceLocation",
                            "value=" + workspaceDir);

        component.addOperation("Settings",
                           "path=" + installer.value("QtCreatorInstallerSettingsFile"),
                           "method=add_array_value",
                           "key=CMakeSpecificSettings/PackageManagerAutoSetup",
                           "value=false")
                            
}

Component.prototype.createOperationsForArchive = function(archive)
{
    component.addOperation("Extract", archive, "@TargetDir@/sdk");

    console.log("Extract archive: " + archive);
    var docker = installer.value("dockerPath");

    component.addOperation("Execute", [docker, "load", "--input", "@TargetDir@/sdk/" + archiveVersion,
                                       "UNDOEXECUTE",
                                       docker, "image", "rm", getDockerTag(), "--force"]);
    if (installer.value("os") === "x11") { // rename image:5.2.0 to image-user:5.2.0 and remove image:5.2.0
        component.addOperation("Execute", [docker, "tag", getDockerTag(false), getDockerTag()]);
        component.addOperation("Execute", [docker, "rmi", getDockerTag(false)]);
    }
    component.addOperation("Delete", "@TargetDir@/" + archiveVersion);
}

function createWrappers(homeDir, installDir, releaseName, wrappersDir)
{
    if (installer.value("os") === "win") {
        component.addOperation("Execute", "powershell", installDir + "/share/qtcreator/apptool/create_wrappers.ps1",
            "-HomeDir", homeDir, "-InstallDir", installDir, "-DockerTag", getDockerTag(), "-Release", releaseName, "-WrappersDir", wrappersDir);
    } else if (installer.value("os") === "mac") {
        component.addOperation("Execute", [installDir + "/Qt\ Creator.app/Contents/Resources/apptool/create_wrappers.sh",
            homeDir, installDir, getDockerTag(), releaseName]);
    } else {
        component.addOperation("Execute", [installDir + "/share/qtcreator/apptool/create_wrappers.sh",
            homeDir, installDir, getDockerTag(), releaseName]);
    }

    component.registerPathForUninstallation(wrappersDir, true);
}

function createKits(installDir, releaseName, wrappersDir)
{
    var wrappersDir = installDir + "/sdk/" + releaseName + "/tools";

    var ideConfigPath = installDir + "/share/qtcreator/@ConfigVariant@/qtcreator";

    var idSuffix = "Aurora-" + releaseName;

    //armv7hl
    create_debugger("armv7hl", installDir + "/bin/gdb-armv7hl-meego-linux-gnueabi",
                    "arm-linux-generic-elf-32bit", idSuffix + ":armv7hl", ideConfigPath, releaseName);
    create_gcc("armv7hl", "arm-linux-generic-elf-32bit", idSuffix + ":armv7hl:gcc", releaseName, wrappersDir, ideConfigPath);
    create_gcc_c("armv7hl", "arm-linux-generic-elf-32bit", idSuffix + ":armv7hl:gcc_c", releaseName, wrappersDir, ideConfigPath);
    create_qmake("armv7hl", idSuffix + ":armv7hl", releaseName, wrappersDir, ideConfigPath);
    create_cmake("armv7hl", idSuffix + ":armv7hl", releaseName, wrappersDir, ideConfigPath);

    create_kit(idSuffix + ":armv7hl",
            idSuffix + ":armv7hl:gcc",
            idSuffix + ":armv7hl:gcc_c",
            idSuffix + ":armv7hl",
            idSuffix + ":armv7hl",
            releaseName + "-armv7hl",
            "QmakeProjectManager.ToolChain.Mer:" + releaseName + "_armv7hl",
            "armv7hl-meego-linux-gnueabi",
            releaseName, ideConfigPath);
    //create_cmake

    //aarch64
    create_debugger("aarch64", installDir + "/bin/gdb-aarch64-meego-linux-gnu",
                    "arm-linux-generic-elf-64bit", idSuffix + ":aarch64", ideConfigPath, releaseName);
    create_gcc("aarch64", "arm-linux-generic-elf-64bit", idSuffix + ":aarch64:gcc", releaseName, wrappersDir, ideConfigPath);
    create_gcc_c("aarch64", "arm-linux-generic-elf-64bit", idSuffix + ":aarch64:gcc_c", releaseName, wrappersDir, ideConfigPath);
    create_qmake("aarch64", idSuffix + ":aarch64", releaseName, wrappersDir, ideConfigPath);
    create_cmake("aarch64", idSuffix + ":aarch64", releaseName, wrappersDir, ideConfigPath);

    create_kit(idSuffix + ":aarch64",
            idSuffix + ":aarch64:gcc",
            idSuffix + ":aarch64:gcc_c",
            idSuffix + ":aarch64",
            idSuffix + ":aarch64",
            releaseName + "-aarch64",
            "QmakeProjectManager.ToolChain.Mer:" + releaseName + "_aarch64",
            "aarch64-meego-linux-gnu",
            releaseName, ideConfigPath);

    //x86-64
    create_debugger("x86_64", installDir + "/bin/gdb-x86_64-meego-linux-gnu",
                    "x86-linux-generic-elf-64bit", idSuffix + ":x86_64", ideConfigPath, releaseName);
    create_gcc("x86_64", "x86-linux-generic-elf-64bit", idSuffix + ":x86_64:gcc", releaseName, wrappersDir, ideConfigPath);
    create_gcc_c("x86_64", "x86-linux-generic-elf-64bit", idSuffix + ":x86_64:gcc_c", releaseName, wrappersDir, ideConfigPath);
    create_qmake("x86_64", idSuffix + ":x86_64", releaseName, wrappersDir, ideConfigPath);
    create_cmake("x86_64", idSuffix + ":x86_64", releaseName, wrappersDir, ideConfigPath);

    create_kit_x86_64(idSuffix + ":x86_64",
    idSuffix + ":x86_64:gcc",
    idSuffix + ":x86_64:gcc_c",
    idSuffix + ":x86_64",
    idSuffix + ":x86_64",
    releaseName + "-x86_64",
    "QmakeProjectManager.ToolChain.Mer:" + releaseName + "_x86_64",
    "x86_64-meego-linux-gnu",
    releaseName, ideConfigPath);
}

function create_debugger(gdbName, gdbExe, gdbAbi, id, ideConfigPath, version) {
    var sdkTool = installer.value("SDKToolBinary");
    var extention = installer.value("os") === "win" ? ".exe" : "";

    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
                                        "addDebugger",
                                        "--id", id,
                                        "--name", "AuroraOS-" + version + "-" + gdbName,
                                        "--engine", 1,
                                        "--binary", gdbExe + extention,
                                        "--abis", gdbAbi],
                            "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmDebugger", "--id", id]);
}

function create_gcc(gccName, gccAbi, id, version, wrappersDir, ideConfigPath) {
    var sdkTool = installer.value("SDKToolBinary");
    var extention = installer.value("os") === "win" ? ".cmd" : "";

    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
    "addTC",
    "--id", "QmakeProjectManager.ToolChain.Mer:" + id,
    "--language", "C++",
    "--name", "AuroraOS-" + version + "-" + gccName,
    "--path", wrappersDir + "/g++-" + gccName + extention,
    "--abi", gccAbi,
    "--language", 2],
    "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmTC", "--id", "QmakeProjectManager.ToolChain.Mer:" + id]);
}

function create_gcc_c(gccName, gccAbi, id, version, wrappersDir, ideConfigPath) {
    var sdkTool = installer.value("SDKToolBinary");
    var extention = installer.value("os") === "win" ? ".cmd" : "";

    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
    "addTC",
    "--id", "QmakeProjectManager.ToolChain.Mer:" + id,
    "--language", "C",
    "--name", "AuroraOS-" + version + "-" + gccName,
    "--path", wrappersDir + "/gcc-" + gccName + extention,
    "--abi", gccAbi,
    "--language", 1],
    "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmTC", "--id", "QmakeProjectManager.ToolChain.Mer:" + id]);
}

function create_qmake(qmakeName, id, version, wrappersDir, ideConfigPath) {
    var sdkTool = installer.value("SDKToolBinary");
    var extention = installer.value("os") === "win" ? ".bat" : "";

    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
    "addQt",
    "--id", id,
    "--name", "AuroraOS-" + version + "-" + qmakeName,
    "--qmake", wrappersDir + "/qmake-" + qmakeName + extention,
    "--type", "RemoteLinux.EmbeddedLinuxQt"],
    "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmQt", "--id", id]);
}

function create_cmake(cmakeName, id, version, wrappersDir, ideConfigPath) {
    var sdkTool = installer.value("SDKToolBinary");
    var extention = installer.value("os") === "win" ? ".bat" : "";

    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
    "addCMake",
    "--id", id,
    "--name", "AuroraOS-" + version + "-" + cmakeName,
    "--path", wrappersDir + "/cmake-" + cmakeName + extention,
    "AutoCreateBuildDirectory", "bool:true"],
    "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmCMake", "--id", id]);
}

function create_kit(gdbId, gccId, gcc_CId, qmakeId, cmakeId, kitName, id, archName, version, ideConfigPath) {
    var sdkTool = installer.value("SDKToolBinary");
    var installDir = toNativePath(installer.value("TargetDir"));
    var targetsDir = toNativePath(installDir + "/sdk/" + releaseName + "/targets");

    var pkg_config_sysroot = "PKG_CONFIG_SYSROOT_DIR=";
    if (installer.value("os") !== "win") {
        pkg_config_sysroot = "PKG_CONFIG_SYSROOT_DIR=" + targetsDir + "/" + archName;
    }
    
    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
    "addKit",
    "--id", id,
    "--name", "AuroraOS-" + version + "-" + kitName,
    "--debuggerid", gdbId,
    "--devicetype", "Mer.Device.Type",
    "--Cxxtoolchain", "QmakeProjectManager.ToolChain.Mer:" + gccId,
    "--Ctoolchain", "QmakeProjectManager.ToolChain.Mer:" + gcc_CId,
    "--qt", qmakeId,
    "--cmake", cmakeId,
    "--cmake-generator", "Ninja",
    "--env", "PKG_CONFIG_DIR=",
    "--env", "PKG_CONFIG_LIBDIR=" + toNativePath(targetsDir + "/" + archName + "/usr/lib/pkgconfig"),
    "--env", pkg_config_sysroot,
    "--cmake-config", "CMAKE_PREFIX_PATH:STRING=%{Qt:QT_INSTALL_PREFIX}",
    "--cmake-config", "QT_QMAKE_EXECUTABLE:STRING=%{Qt:qmakeExecutable}"],
    "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmKit", "--id", id]);
}

function create_kit_x86_64(gdbId, gccId, gcc_CId, qmakeId, cmakeId, kitName, id, archName, version, ideConfigPath) {
    var sdkTool = installer.value("SDKToolBinary");
    var installDir = toNativePath(installer.value("TargetDir"));
    var targetsDir = toNativePath(installDir + "/sdk/" + releaseName + "/targets");

    var pkg_config_sysroot = "PKG_CONFIG_SYSROOT_DIR=";
    if (installer.value("os") !== "win") {
        pkg_config_sysroot = "PKG_CONFIG_SYSROOT_DIR=" + targetsDir + "/" + archName;
    }

    component.addOperation("Execute", ["{0,2,3}", sdkTool, "--sdkpath=" + ideConfigPath,
    "addKit",
    "--id", id,
    "--name", "AuroraOS-" + version + "-" + kitName,
    "--debuggerid", gdbId,
    "--devicetype", "Mer.Device.Type",
    "--Cxxtoolchain", "QmakeProjectManager.ToolChain.Mer:" + gccId,
    "--Ctoolchain", "QmakeProjectManager.ToolChain.Mer:" + gcc_CId,
    "--qt", qmakeId,
    "--cmake", cmakeId,
    "--cmake-generator", "Ninja",
    "--env", "PKG_CONFIG_DIR=",
    "--env", "PKG_CONFIG_LIBDIR=" + toNativePath(targetsDir + "/" + archName + "/usr/lib/pkgconfig"),
    "--env", pkg_config_sysroot,
    "--cmake-config", "CMAKE_PREFIX_PATH:STRING=%{Qt:QT_INSTALL_PREFIX}",
    "--cmake-config", "QT_QMAKE_EXECUTABLE:STRING=%{Qt:qmakeExecutable}"],
    "UNDOEXECUTE", [sdkTool, "--sdkpath=" + ideConfigPath, "rmKit", "--id", id]);
}
