/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef AURORA_MDM_CONNECTIVITY_VPNMANAGER_H
#define AURORA_MDM_CONNECTIVITY_VPNMANAGER_H

#include <QObject>
#include <QScopedPointer>
#include <QStringList>
#include <QVariantMap>
#include <QString>
#include <QList>
#include <QSharedPointer>

namespace Aurora {
namespace Mdm {
namespace Connectivity {

class VpnProfilePrivate;
class VpnManager;
class Q_DECL_EXPORT VpnProfile : public QObject
{
    Q_OBJECT

    Q_ENUMS(State)
    Q_PROPERTY(QString name READ name WRITE setName NOTIFY nameChanged)
    Q_PROPERTY(QString host READ host WRITE setHost NOTIFY hostChanged)
    Q_PROPERTY(QString domain READ domain WRITE setDomain NOTIFY domainChanged)
    Q_PROPERTY(QString type READ type WRITE setType NOTIFY typeChanged)
    Q_PROPERTY(bool defaultRoute READ defaultRoute WRITE setDefaultRoute NOTIFY defaultRouteChanged)
    Q_PROPERTY(bool autoConnect READ autoConnect WRITE setAutoConnect NOTIFY autoConnectChanged)
    Q_PROPERTY(QVariantMap providerProperties READ providerProperties NOTIFY providerPropertiesChanged)
    Q_PROPERTY(State state READ state NOTIFY stateChanged)
    Q_PROPERTY(bool valid READ valid NOTIFY validChanged)

public:
    enum Error {
        Success = 0,
        InternalError,
    };
    Q_ENUM(Error);

    enum State {
        Idle,
        Failure,
        Association,
        Configuration,
        Ready,
        Disconnect,
        Online
    };
    Q_ENUM(State);

    ~VpnProfile() override;

    QString name() const;
    QString host() const;
    QString domain() const;
    QString type() const;
    bool defaultRoute() const;
    bool autoConnect() const;
    QVariantMap providerProperties() const;
    State state() const;
    bool valid() const;

    void setName(const QString &value);
    void setHost(const QString &value);
    void setDomain(const QString &value);
    void setType(const QString &value);
    void setDefaultRoute(bool value);
    void setAutoConnect(bool value);
    void setProviderProperty(const QString &name, const QVariant &value);

    void connect();
    void disconnect();
    void remove();

signals:
    void nameChanged();
    void hostChanged();
    void domainChanged();
    void typeChanged();
    void defaultRouteChanged();
    void autoConnectChanged();
    void providerPropertiesChanged();
    void stateChanged();
    void validChanged();

    void setNameFinished(Error result);
    void setHostFinished(Error result);
    void setDomainFinished(Error result);
    void setTypeFinished(Error result);
    void setDefaultRouteFinished(Error result);
    void setAutoConnectFinished(Error result);
    void setProviderPropertyFinished(const QString &name, Error result);

    void connectFinished(Error result);
    void disconnectFinished(Error result);
    void removeFinished(Error result);

private:
    VpnProfile(const QString &path, VpnManager *manager);

    friend class VpnManager;
    friend class VpnManagerPrivate;
    friend class VpnProfilePrivate;

    Q_DISABLE_COPY(VpnProfile)
    QScopedPointer<VpnProfilePrivate> d_ptr;
};

class VpnManagerPrivate;
class Q_DECL_EXPORT VpnManager : public QObject
{
    Q_OBJECT

public:
    enum Error {
        Success = 0,
        InternalError,
    };
    Q_ENUM(Error);

    struct CreateParams {
        QString name;
        QString host;
        QString domain;
        QString type;
        QVariantMap providerProperties;
    };

    VpnManager(QObject *parent = nullptr);
    ~VpnManager() override;

    // returns transaction id
    quint64 createVpn(const CreateParams &params);
    QList<QSharedPointer<VpnProfile>> vpns();

signals:
    void createVpnFinished(quint64 transactionId, QSharedPointer<VpnProfile> vpn, Error result);
    void newVpn(QSharedPointer<VpnProfile> vpn);

private:
    friend class VpnManagerPrivate;
    friend class VpnProfile;
    friend class VpnProfilePrivate;

    Q_DISABLE_COPY(VpnManager)
    QScopedPointer<VpnManagerPrivate> d_ptr;
};

} // namespace Connectivity
} // namespace Mdm
} // namespace Aurora

#endif // AURORA_MDM_CONNECTIVITY_VPNMANAGER_H
