// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: Proprietary

#ifndef INAPPTABBASE_H
#define INAPPTABBASE_H

#include "iinapptab.h"

#include <QDBusConnection>
#include <QDBusInterface>
#include <QFileSystemWatcher>
#include <QLoggingCategory>
#include <QProcess>
#include <QQmlParserStatus>
#include <QQuickWindow>
#include <QScopedPointer>
#include <QTemporaryDir>

Q_DECLARE_LOGGING_CATEGORY(inapptabLog)

/**
 * @brief Enum specifying which accessibility mode to use.
 */
enum ActionListType { Base, Advanced };

/**
 * @brief Base class for in-app tab process management and D-Bus setup.
 */
class InAppTabBase : public IInAppTab, public QQmlParserStatus
{
    Q_OBJECT
    Q_INTERFACES(QQmlParserStatus)

public:
    explicit InAppTabBase(ActionListType type, QObject *parent = nullptr);
    ~InAppTabBase();

    /**
   * @brief Requests to open the specified URL.
   * Subclasses must implement sendOpenRequest() to call the correct D-Bus
   * method.
   * @param url The URL to load.
   */
    Q_INVOKABLE void open(const QString &url);

    void classBegin() override;
    void componentComplete() override;

protected:
    /**
   * @brief Called after D-Bus connection is established.
   * Subclasses must implement this to send the initial request.
   * @param url The URL to load.
   */
    virtual void sendOpenRequest(const QString &url, bool advancedAccessibility) = 0;

    /**
   * @brief Subclass can connect extra signals (e.g., Finished).
   */
    virtual void connectExtraSignals();

    bool m_advancedAccessibility;
    QDBusConnection m_connection;
    QScopedPointer<QDBusInterface> m_interface;

private:
    QScopedPointer<QProcess> m_process;
    QScopedPointer<QTemporaryDir> m_tempDir;
    QString m_socketPath;
    QString m_lastUrl;
    bool m_connectionEstablished = false;
    QFileSystemWatcher *m_socketWatcher = nullptr;
    QString m_socketDir;
    QQuickWindow *m_window = nullptr;
    bool m_deviceIsLocked = false;

    void tryConnectToPeer();
    void establishConnection(const QString &url);
    void watchWindowLifecycle();
    void resumeInteraction();
    void hideInteraction();
    void watchSocketDirectory();

private slots:
    void onLoadStart(const QString &url);
    void onLoadEnd(const QString &url);
    void onLoadError(const QString &url, const QString &error);
    void onClosed();
    void onHideApp();
    void onDeviceLockChanged(int state);
};

#endif // INAPPTABBASE_H
