/*
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>

#include <QObject>

#include "jsdialogrequest.h"

namespace Aurora {
namespace WebView {

class ConfirmDialogRequestInternal;

/**
 * @class ConfirmDialogRequest
 * @brief Represents JavaScript confirm dialog request
 * 
 * You can use it to handle a JS confirm dialog
 */
class ConfirmDialogRequest : public JSDialogRequest
{
    Q_GADGET

    /// URL of the web page that called the function
    Q_PROPERTY(QString sourceUrl READ sourceUrl)
    /// message from the function that triggered the dialog
    Q_PROPERTY(QString message READ message)
    /// returns true if the request is valid, do not use otherwise
    Q_PROPERTY(bool isValid READ isValid)
public:
    /**
     * @brief Constructs a new ConfirmDialogRequest object
     * This constructor is declared to register
     * this class in QML
     */
    ConfirmDialogRequest();
    ConfirmDialogRequest(const ConfirmDialogRequest &other) = default;
    /**
     * @brief Destroys the ConfirmDialogRequest object
     */
    ~ConfirmDialogRequest() = default;

    /**
     * @brief Used to send true to confirm JavaScript function and return to the web page
     */
    Q_INVOKABLE virtual void ok();
    /**
     * @brief Used to send false to confirm JavaScript function and return to the web page
     */
    Q_INVOKABLE virtual void cancel();

    bool isValid() const override;

private:
    ConfirmDialogRequest(
        QString originUrl,
        QString messageText,
        std::shared_ptr<ConfirmDialogRequestInternal> dialogRequestInternal);

    std::shared_ptr<ConfirmDialogRequestInternal> m_d;

    friend class ConfirmDialogRequestInternal;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::ConfirmDialogRequest)
