/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>

#include <QString>

#include "downloaditem.h"

namespace Aurora {
namespace WebView {

/**
 * @class IDownloadHandler
 * @brief Interface for DownloadHandler.
 * 
 * You can implement this interface to override default DownloadHandler. However,
 * note that instantiating an external implementation disables default QML wrapper,
 * so you might also need to create it on your own.
 *
 * Browser engine starts using an external implementation as soon as it is created.
 * Please note that only the last created implementation is used. As soon as
 * a new implementation is instantiated, all previous implementations become invalid.
 */
class IDownloadHandler
{
public:
    /**
     * @brief Construct a new IDownloadHandler object
     */
    IDownloadHandler();
    /**
     * @brief Destroy the IDownloadHandler object
     */
    virtual ~IDownloadHandler() = default;

    /**
     * @brief checks if download requests are allowed.
     * 
     * @return return true to inidicate that download requests are allowed.
     */
    virtual bool downloadRequestsAllowed() = 0;
    /**
     * @brief sets download request behavior
     * 
     * @param allowed set allowed to false to decline all requests automatically.
     */
    virtual void setDownloadRequestsAllowed(bool allowed) = 0;
    /**
     * @brief notifyDownloadRequested is called when a new download starts (e.g., when the user clicks a link to download a file).
     * 
     * @param downloadItem is an object that can be used to accept or cancel the download.
     * @param suggestedName is a suggested name for the file that is to be downloaded.
     */
    virtual void notifyDownloadRequested(const DownloadItem &downloadItem, QString suggestedName)
        = 0;
    /**
     * @brief this method is called on each download update.
     * 
     * @param downloadItem can be used to get the state of the download and interact with it.
     */
    virtual void notifyDownloadUpdated(const DownloadItem &downloadItem) = 0;
};

} // namespace WebView
} // namespace Aurora
