/*******************************************************************************
 *
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 *
 ******************************************************************************/

#ifndef RM_LIB_TYPES_H
#define RM_LIB_TYPES_H

#include <QMetaType>
#include <QString>
#include <QVariantMap>
#include <QVector>
#include <sys/types.h>

namespace RuntimeManager {

/**
 * @brief Describes the start reason for an application or a background task
 */
enum class StartReason {
    /** The start reason is undefined */
    Unknown = 0,
    /** The application has been explicitly requested to start by the user */
    Manual,
    /** The application or task was automatically started at the start of the
     * user session */
    Autostart,
    /** The application was started as a intent handler */
    Intent,
    /** The application was started by a MDM request */
    MDM,
    /** The application or task was started by a system service */
    System,
};

/**
 * @brief Describes how an application or task was stopped
 */
enum class StopReason {
    /** The stop reason is undefined. This also captures the case when the
     * application exited spontaneously */
    Unknown = 0,
    /** The application has been explicitly requested to stop by the user */
    Manual,
    /** The application has been explicitly killed by a system service */
    System = 5,
    /** The application killed by the OOM */
    OOM,
};

/**
 * @brief Describes the application or background task's state
 */
enum ApplicationState {
    /** The application has been asked to start, but it's still in the loading
     * or sandboxing setup stage */
    Starting = 0,
    /** The application has at least one window visible and focused */
    Active,
    /** The application has a visible window, but not currently focused */
    Inactive,
    /** None of the application windows are currently visible. The application
     * processes might be frozen. */
    Hidden,
    /** The application has been asked to terminate */
    Terminating,
    /** The application is not currently running */
    NotRunning,
    Invalid = 255,
};

/**
 * @brief Describes the state of an application
 */
struct RunningApplication
{
    /** The application ID */
    QString appInstanceId;
    /** PID of the application's main process (this is typically the process
     * managing the sandbox) */
    pid_t pid;
    /** Start reason for the application */
    StartReason startReason;
    /** Current state of the application */
    ApplicationState state;
    /** Free-form data, for future use */
    QVariantMap details;
};

/**
 * @brief Describes the state of a background task
 */
struct RunningService
{
    /** The ID of the application owning the task */
    QString appId;
    /** The name of the task (as given to the Task constructor) */
    QString name;
    /** PID of the task's main process (this is typically the process managing
     * the sandbox) */
    pid_t pid;
    /** Start reason for the task */
    StartReason startReason;
    /** Free-form data, for future use */
    QVariantMap details;
};

} // namespace RuntimeManager

Q_DECLARE_METATYPE(RuntimeManager::RunningApplication);
Q_DECLARE_METATYPE(RuntimeManager::RunningService);

#endif // RM_LIB_TYPES_H
