/*
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "certificate.h"
#include "enums/types.h"

#include <QObject>
#include <QString>
#include <QVariant>

#include <memory>

namespace Aurora {
namespace WebView {

class ClientCertificateRequestInternal;

/**
 * @class ClientCertificateRequest
 * @brief Contains additional informaton about client certificate request.
 */
class ClientCertificateRequest
{
    Q_GADGET

    /// Hostname and port of the SSL server
    Q_PROPERTY(QString host READ host CONSTANT)

    /// Port of the SSL server
    Q_PROPERTY(qint32 port READ port CONSTANT)
    /// List of available certificates
    Q_PROPERTY(QVariantList certificates READ certificatesQML CONSTANT)
    /// true if this request is valid (do not use otherwise)
    Q_PROPERTY(bool isValid READ isValid)
public:
    /**
     * @brief Constructs a new ClientCertificateRequest object
     * This constructor is declared to register
     * this class in QML
     */
    ClientCertificateRequest();
    /**
     * @brief Destroys the ClientCertificateRequest object
     */
    virtual ~ClientCertificateRequest();

    /**
     * @brief this method should be called to select client certificate
     * by its ID. If the ID is invalid, the request is cancelled.
     *
     * @param certificateId certificate ID.
     */
    Q_INVOKABLE virtual void select(qint32 certificateId);
    /**
     * @brief this method should be called to cancel the request.
     */
    Q_INVOKABLE virtual void cancel();

    QString host() const;
    qint32 port() const;
    QList<Certificate> certificates() const;
    bool isValid() const;

private:
    ClientCertificateRequest(
        const QString &host,
        qint32 port,
        QList<Certificate> certificates,
        std::shared_ptr<ClientCertificateRequestInternal> certificateRequestInternal);

    const QString m_host;
    const quint32 m_port;
    const QList<Certificate> m_certificates;

    QVariantList certificatesQML() const;

    std::shared_ptr<ClientCertificateRequestInternal> m_certificateRequestInternal;

    friend class ClientCertificateRequestInternal;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::ClientCertificateRequest)
