/*******************************************************************************
 *
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 *
 ******************************************************************************/

#ifndef RM_LIB_SCHEDULE_H
#define RM_LIB_SCHEDULE_H

#include "global.h"

#include <QDateTime>
#include <QList>
#include <QVariantMap>

namespace RuntimeManager {

class Task;

/**
 * @brief Describe the scheduling of a background task
 * @class Schedule schedule.h <RuntimeManager/Schedule>
 *
 * The Schedule class represents the schedule of a background task.
 */
class RM_EXPORT Schedule
{
public:
    /**
     * Constructor.
     */
    Schedule();
    Schedule(const Schedule &other);
    ~Schedule();

    /**
     * Set the start date for scheduled events.
     */
    Schedule &sinceDate(const QDateTime &startDate);
    /**
     * Get the start date for scheduled events.
     *
     * @return the start date of the schedule.
     */
    QDateTime startDate() const;

    /**
     * Set the end date for scheduled events.
     */
    Schedule &tillDate(const QDateTime &endDate);
    /**
     * Get the end date for scheduled events.
     *
     * @return the end date of the schedule.
     */
    QDateTime endDate() const;

    /**
     * Set the months (as indexes from 0 to 11) during which the task should be
     * triggered. If not set, the task will trigger on any month.
     */
    Schedule &onMonths(const QList<int> &months);
    /**
     * Get the months for the schedule.
     *
     * @return the months for the schedule.
     */
    QList<int> months() const;

    /**
     * Set the week days (as indexes from 0 to 6, 0 being Sunday) during which
     * the task should be triggered. If not set, the task will trigger on any
     * day.
     */
    Schedule &onWeekDays(const QList<int> &days);
    /**
     * Get the days for the schedule.
     *
     * @return the days for the schedule.
     */
    QList<int> weekDays() const;

    /**
     * Set the hours (from 0 to 23) during which the task should be triggered.
     * If not set, the task will trigger on any hour.
     */
    Schedule &onHours(const QList<int> &hours);
    /**
     * Get the hours for the schedule.
     *
     * @return the hours for the schedule.
     */
    QList<int> hours() const;

    /**
     * Set the minutes (from 0 to 59) during which the task should be
     * triggered. If not set, the task will trigger on any minute.
     */
    Schedule &onMinutes(const QList<int> &minutes);
    /**
     * Get the minutes for the schedule.
     *
     * @return the minutes for the schedule.
     */
    QList<int> minutes() const;

private:
    Schedule(const QVariantMap &fieldMap);

private:
    friend class Task;
    QVariantMap m_fields;
};

} // namespace RuntimeManager

#endif // RM_LIB_SCHEDULE_H
