/*
 * Copyright (C) 2006 Apple Computer, Inc.
 * Copyright (C) 2007 Nikolas Zimmermann <zimmermann@kde.org>
 * Copyright (C) Research In Motion Limited 2010-2012. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef THIRD_PARTY_BLINK_RENDERER_CORE_LAYOUT_SVG_LAYOUT_SVG_TEXT_H_
#define THIRD_PARTY_BLINK_RENDERER_CORE_LAYOUT_SVG_LAYOUT_SVG_TEXT_H_

#include "third_party/blink/renderer/core/layout/svg/layout_svg_block.h"

namespace blink {

class LayoutSVGInlineText;

class LayoutSVGText final : public LayoutSVGBlock {
 public:
  explicit LayoutSVGText(Element*);
  ~LayoutSVGText() override;
  void Trace(Visitor*) const override;

  bool IsChildAllowed(LayoutObject*, const ComputedStyle&) const override;

  void SetNeedsPositioningValuesUpdate() {
    NOT_DESTROYED();
    needs_positioning_values_update_ = true;
  }
  void SetNeedsTextMetricsUpdate() {
    NOT_DESTROYED();
    needs_text_metrics_update_ = true;
  }
  gfx::RectF VisualRectInLocalSVGCoordinates() const override;
  gfx::RectF ObjectBoundingBox() const override;
  gfx::RectF StrokeBoundingBox() const override;
  bool IsObjectBoundingBoxValid() const;

  void AddOutlineRects(Vector<PhysicalRect>&,
                       OutlineInfo*,
                       const PhysicalOffset& additional_offset,
                       NGOutlineType) const override;

  // These two functions return a LayoutSVGText, a LayoutNGSVGText, or nullptr.
  static LayoutSVGBlock* LocateLayoutSVGTextAncestor(LayoutObject*);
  static const LayoutSVGBlock* LocateLayoutSVGTextAncestor(const LayoutObject*);

  static void NotifySubtreeStructureChanged(LayoutObject*,
                                            LayoutInvalidationReasonForTracing);

  bool NeedsReordering() const {
    NOT_DESTROYED();
    return needs_reordering_;
  }
  const HeapVector<Member<LayoutSVGInlineText>>& DescendantTextNodes() const {
    NOT_DESTROYED();
    return descendant_text_nodes_;
  }

  void RecalcVisualOverflow() override;

  const char* GetName() const override {
    NOT_DESTROYED();
    return "LayoutSVGText";
  }

 private:
  bool RespectsCSSOverflow() const override {
    NOT_DESTROYED();
    return false;
  }

  bool IsOfType(LayoutObjectType type) const override {
    NOT_DESTROYED();
    return type == kLayoutObjectSVGText || LayoutSVGBlock::IsOfType(type);
  }

  void Paint(const PaintInfo&) const override;
  bool NodeAtPoint(HitTestResult&,
                   const HitTestLocation&,
                   const PhysicalOffset& accumulated_offset,
                   HitTestAction) override;
  PositionWithAffinity PositionForPoint(const PhysicalOffset&) const override;

  void UpdateLayout() override;

  void AbsoluteQuads(Vector<gfx::QuadF>&,
                     MapCoordinatesFlags mode = 0) const override;

  void AddChild(LayoutObject* child,
                LayoutObject* before_child = nullptr) override;
  void RemoveChild(LayoutObject*) override;

  void StyleDidChange(StyleDifference, const ComputedStyle* old_style) override;
  void WillBeDestroyed() override;

  RootInlineBox* CreateRootInlineBox() override;

  void SubtreeStructureChanged(LayoutInvalidationReasonForTracing);
  void UpdateTransformAffectsVectorEffect();

  bool needs_reordering_ : 1;
  bool needs_positioning_values_update_ : 1;
  bool needs_text_metrics_update_ : 1;
  HeapVector<Member<LayoutSVGInlineText>> descendant_text_nodes_;
};

template <>
struct DowncastTraits<LayoutSVGText> {
  static bool AllowFrom(const LayoutObject& object) {
    return object.IsSVGText();
  }
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_CORE_LAYOUT_SVG_LAYOUT_SVG_TEXT_H_
