/*
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
*/
import QtQuick 2.6
import Sailfish.Silica 1.0
import Nemo.DBus 2.0

ApplicationWindow {
    id: root

    property ShareSystemDialog _shareDialog
    property string senderService
    property string senderUuid

    function _open(shareActionConfiguration) {
        if (_shareDialog) {
            _shareDialog.lower()
            _shareDialog.destroy()
        }
        _shareDialog = shareDialogComponent.createObject(
                    root, { "shareActionConfiguration": shareActionConfiguration })
        autoDismissDelay.restart()
    }

    initialPage: Component {
        Page {
            allowedOrientations: Orientation.All
        }
    }
    allowedOrientations: Orientation.All
    _defaultPageOrientations: Orientation.All
    _defaultLabelFormat: Text.PlainText

    Component {
        id: shareDialogComponent

        ShareSystemDialog {
            onRequestFiles: {
                dbusAdaptor.emitTargetedSignal(root.senderService, "convertToFile", root.senderUuid)
            }

            onClosed: {
                autoDestroy.start()
            }

            Component.onCompleted: {
                if (!autoDestroy.running) {
                    activate()
                }
            }
        }
    }

    DBusInterface {
        bus: DBus.SystemBus
        service: 'com.nokia.mce'
        path: '/com/nokia/mce/signal'
        iface: 'com.nokia.mce.signal'
        signalsEnabled: true

        function display_status_ind(state) {
            if (state !== "on" && !!_shareDialog) {
                _shareDialog.dismiss()
                autoDestroy.start()
            }
        }
    }

    Timer {
        id: autoDismissDelay

        interval: 1000

        onTriggered: {
            if (_shareDialog) {
                _shareDialog.autoDismiss = true
            }
        }
    }

    Timer {
        id: autoDestroy

        // Wait a good amount of time before auto-exiting. Otherwise, if sharing triggers launching
        // of an app via dbus, and sailfish-share quits before the app is launched, dbus will abort
        // launching of that app.
        interval: 30*1000

        onTriggered: {
            console.warn("sailfish-share: exiting...")
            Qt.quit()
        }
    }

    DBusAdaptor {
        id: dbusAdaptor

        service: "org.sailfishos.share"
        path: "/"
        iface: "org.sailfishos.share"

        function share(shareActionConfiguration) {
            root.senderService = dbusAdaptor.senderService()
            root.senderUuid = shareActionConfiguration.uuid
            autoDestroy.stop()
            root._open(shareActionConfiguration)
        }

        function fileReady(shareActionConfiguration) {
            if(_shareDialog) {
                _shareDialog.shareActionNewConfiguration = shareActionConfiguration
                _shareDialog.fileConvertError = false
                _shareDialog.waitingForFile = false
                _shareDialog.setShareMethodLoaderSource()
            }
        }

        function fileConvertError() {
            if(_shareDialog) {
                //% "An error has occurred"
                _shareDialog.createErrorNotice(qsTrId("sailfishshare-nt-convert-error"))
                _shareDialog.fileConvertError = true
                _shareDialog.waitingForFile = false
            }
        }
    }
}
