/****************************************************************************************
**
** SPDX-FileCopyrightText: 2014 Jolla Ltd.
** SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: BSD 3-Clause
**
****************************************************************************************/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import QtGraphicalEffects 1.0

Item {
    id: root

    property Flickable flickable
    property bool flicking
    property bool inBounds: (!flickable.pullDownMenu || !flickable.pullDownMenu.active) && (!flickable.pushUpMenu || !flickable.pushUpMenu.active)
    property bool pressed: scrollUpButton.pressed || scrollDownButton.pressed
    property bool active: visibilityTimer.running || pressed
    property bool clicked
    property int directionsEnabled: QuickScrollDirection.UpAndDown
    readonly property bool _isVisible: gradient.opacity > 0.0 ||
                                       scrollUpButton.opacity > 0.0 ||
                                       scrollDownButton.opacity > 0.0
    readonly property bool _atBound: flickable.atYBeginning || flickable.atYEnd

    function _buttonClicked() {
        clicked = true

        if (active) {
            visibilityTimer.interval = 100
            visibilityTimer.restart()
        }
    }

    on_AtBoundChanged: {
        if (visibilityTimer.running && !clicked) {
            visibilityTimer.interval = 3000 - new Date().getTime() + visibilityTimer.startTime
            visibilityTimer.restart()
        }
    }

    onActiveChanged: {
        if (active) {
            gradientOpacityAnimation.duration = 300
            gradient.opacity = 1.0
        } else {
            gradientOpacityAnimation.duration = 200
            gradient.opacity = 0.0
        }
    }

    anchors {
        right: parent ? parent.right : undefined
        bottom: parent ? parent.bottom : undefined
        bottomMargin: Theme.dp(200) - Theme.iconSizeSmall
    }

    enabled: active
    implicitHeight: buttonBar.height
    implicitWidth: Theme.itemSizeExtraLarge

    Timer {
        id: visibilityTimer

        property var startTime

        onRunningChanged: {
            if (running) {
                startTime = new Date().getTime()
            }
        }
    }

    Timer {
        id: appearTimer

        onTriggered: {
            if (!visibilityTimer.running && flickable.flickingVertically && Math.abs(flickable.verticalVelocity) > Screen.height && root.inBounds) {
                visibilityTimer.interval = 2000
                visibilityTimer.restart()
            }
        }

        interval: 300
    }

    Connections {
        onDraggingVerticallyChanged: {
            root.clicked = false

            if (!flickable.draggingVertically) {
                if (!visibilityTimer.running) {
                    appearTimer.restart()
                }
            } else {
                appearTimer.stop()
            }

            if (visibilityTimer.running) {
                visibilityTimer.interval = (flickable.atYEnd || flickable.atYBeginning) ? 3000 : 2000
                visibilityTimer.restart()
            }
        }

        target: flickable
    }

    RadialGradient {
        id: gradient

        anchors.verticalCenter: buttonBar.verticalCenter

        width: parent.width
        height: Theme.dp(300)
        opacity: 0.0
        horizontalOffset: Theme.itemSizeExtraLarge * 0.5
        verticalOffset: 0
        horizontalRadius: Theme.itemSizeExtraLarge * 2
        verticalRadius: height
        angle: 0
        gradient: Gradient {
            GradientStop { position: 0.0; color: Theme.colorScheme === Theme.LightOnDark ? Theme.rgba("black", 0.4)
                                                                                         : Theme.rgba("white", 0.6) }
            GradientStop { position: 0.5; color: "transparent" }
        }

        Behavior on opacity {
            FadeAnimation {
                id: gradientOpacityAnimation

                easing.type: Easing.InOutCubic
            }
        }
    }

    Column {
        id: buttonBar

        QuickScrollButton {
            id: scrollUpButton

            onClicked: {
                if (active) {
                    flickable.scrollToTop()
                    root._buttonClicked()
                }
            }

            active: root.active
            direction: QuickScrollDirection.Up
            directionsEnabled: root.directionsEnabled
            source: "image://theme/icon-splus-up-alt"
            flickable: root.flickable
        }

        QuickScrollButton {
            id: scrollDownButton

            onClicked: {
                if (active) {
                    flickable.scrollToBottom()
                    root._buttonClicked()
                }
            }

            active: root.active
            direction: QuickScrollDirection.Down
            directionsEnabled: root.directionsEnabled
            source: "image://theme/icon-splus-down-alt"
            flickable: root.flickable
        }
    }
}
