/****************************************************************************************
**
** SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/
import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as SilicaPrivate
import Sailfish.Silica.Background 1.0
import Sailfish.Lipstick 1.0
import Sailfish.TransferEngine 1.0
import Sailfish.Share 1.0
import Nemo.DBus 2.0
import Sailfish.Policy 1.0
import com.jolla.settings.system 1.0
import org.nemomobile.notifications 1.0

SystemDialog {
    id: root

    autoDismiss: false

    signal requestFiles()

    property var shareActionConfiguration
    onShareActionConfigurationChanged: shareAction.loadConfiguration(shareActionConfiguration)

    property var shareActionNewConfiguration
    onShareActionNewConfigurationChanged: newShareAction.loadConfiguration(shareActionNewConfiguration)

    readonly property int _windowMargin: Theme.paddingMedium
    readonly property int _topWindowMargin: Screen.sizeCategory >= Screen.Large
                                            ? Theme.itemSizeHuge
                                            : (orientation === Qt.PortraitOrientation || orientation === Qt.InvertedPortraitOrientation)
                                              ? Theme.itemSizeHuge
                                              : 0

    readonly property real _windowWidthInPortrait: Screen.sizeCategory >= Screen.Large
                                                   ? Screen.height / 3
                                                   : Screen.width - _windowMargin*2

    readonly property real _windowWidthInLandscape: Screen.sizeCategory >= Screen.Large
                                                    ? Screen.height / 3
                                                    : Screen.height * 3/4 - _windowMargin*2

    property bool waitingForFile: false
    property bool fileConvertError: false

    property string selectedMethodSource
    property var selectedTransferMethodInfo
    property string status: {
        if (fileConvertError)
            return "error"
        if (waitingForFile)
            return "loading"
    }

    Binding on status {
        when: shareMethodLoader.status === Loader.Error
        value: "error"
    }

    Binding on status {
        when: shareMethodLoader.status === Loader.Loading
        value: "loading"
    }

    function setShareMethodLoaderSource() {
        newShareAction.selectedTransferMethodInfo = root.selectedTransferMethodInfo
        if (newShareAction.selectedTransferMethodInfo && newShareAction.selectedTransferMethodInfo.methodId !== "Bluetooth")
            newShareAction.replaceFileResourcesWithFileDescriptors()

        shareMethodLoader.setSource(selectedMethodSource,
                                    { "shareAction": newShareAction })
    }

    function createErrorNotice(errorMessege) {
        Notices.show(errorMessege, Notice.Short, Notice.Bottom)
    }

    contentHeight: content.height
    category: SystemDialogWindow.Alarm

    layoutItem.contentItem.x: (layoutItem.width - layoutItem.contentItem.width) * 0.5
    layoutItem.contentItem.y: layoutItem.height - layoutItem.contentItem.height - _windowMargin
    layoutItem.contentItem.width: (orientation === Qt.PortraitOrientation || orientation === Qt.InvertedPortraitOrientation)
                                  ? _windowWidthInPortrait
                                  : _windowWidthInLandscape
    layoutItem.contentItem.height: content.height

    backgroundRect: calcBackgroundRect()

    function calcBackgroundRect() {
        return __silica_applicationwindow_instance.mapFromItem(layoutItem,
                                                               layoutItem.contentItem.x, layoutItem.contentItem.y,
                                                               layoutItem.contentItem.width, layoutItem.contentItem.height)
    }

    Timer {
        id: updateBackgroundTimer

        property var bindings: {
            // force binding to re-evaluate on orientation change (Portrait -> InvertedPortrait or Landscape -> InvertedLandscape)
            const orientationBindings = orientation
            const safeZoneBindings = SafeZoneRect.insets && SafeZoneRect.appInsets
            const positionBindings = layoutItem.contentItem.x + layoutItem.contentItem.y
                                   + layoutItem.contentItem.width + layoutItem.contentItem.height
            return [orientationBindings, safeZoneBindings, positionBindings]
        }

        interval: 1

        onBindingsChanged: {
            updateBackgroundTimer.start()
        }

        onTriggered: {
            backgroundRect = calcBackgroundRect()
        }
    }

    Component.onCompleted: {
        shareAction.loadConfiguration(root.shareActionConfiguration)
    }

    function containsDataOnly(resources)
    {
        for (var r in resources) {
            if (!resources[r].hasOwnProperty("data")) {
                return false
            }
        }
        return true
    }

    ShareAction {
        id: shareAction

        onDone: root.dismiss()

        function shareError(error, message) {
            root.status = "error"
            console.warn("Failed to share:", error, "with message:", message)
        }

        function disabledByMdm() {
            shareMethodLoader.sourceComponent = mdmBanner
        }
    }

    ShareAction {
        id: newShareAction

        onDone: root.dismiss()

        function shareError(error, message) {
            root.status = "error"
            console.warn("Failed to share:", error, "with message:", message)
        }

        function disabledByMdm() {
            shareMethodLoader.sourceComponent = mdmBanner
        }
    }

    SailfishSharingMethodsModel {
        id: sharingMethodsModel

        mimeTypeFilter: shareAction.mimeType
        filterByMultipleFileSupport: shareAction.resources.length > 1
        containsDataOnly: root.containsDataOnly(shareAction.resources)
    }

    SilicaPrivate.RoundedWindowCorners {
       anchors.fill: parent
       radius: Theme.paddingLarge
    }

    Component {
        id: mdmBanner

        DisabledByMdmBanner {
        }
    }

    SilicaFlickable {
        id: content

        width: parent.width
        height: {
            var screenHeight = (orientation === Qt.PortraitOrientation || orientation === Qt.InvertedPortraitOrientation
                                ? Screen.height
                                : Screen.width) - __silica_applicationwindow_instance.pageStack.panelSize
            return Math.min(screenHeight - _windowMargin*2 - _topWindowMargin, contentHeight)
        }

        contentHeight: {
            if (loadingIndicator.running || delayLoadingIndicator.running) {
                return loadingIndicator.y + loadingIndicator.height
            } else if (shareMethodsColumn.visible) {
                return shareMethodsColumn.y + shareMethodsColumn.height
            } else {
                return shareMethodLoader.y + shareMethodLoader.height
            }
        }
        clip: true

        Behavior on contentHeight {
            NumberAnimation { duration: 150 }
        }

        VerticalScrollDecorator {}

        SystemDialogHeader {
            id: dialogHeader

            width: parent.width
            height: Theme.itemSizeMedium

            Label {
                x: Theme.paddingLarge
                anchors.verticalCenter: parent.verticalCenter

                width: parent.width - x - systemDialogIcon.width

                font.weight: Theme.fontSizeMedium
                text: shareAction.title.length
                      ? shareAction.title
                         //% "Share"
                      : qsTrId("sailfishshare-he-share")
            }

            IconButton {
                id: systemDialogIcon

                anchors {
                    verticalCenter: parent.verticalCenter
                    right: parent.right
                    rightMargin: Theme.dp(4)
                }
                icon.source: "image://theme/icon-splus-cancel"
                onClicked: root.dismiss()
            }

            Rectangle {
                width: parent.width
                height: Theme.dp(1)

                anchors.bottom: parent.bottom

                color: Theme.rgba(Theme.primaryColor, Theme.opacityFaint)
            }
        }

        Loader {
            id: shareMethodLoader

            width: parent.width
            height: status === Loader.Ready ? item.height : placeholder.height

            anchors.top: dialogHeader.bottom

            asynchronous: true

            visible: !shareMethodsColumn.visible
            opacity: visible ? 1 : 0
            Behavior on opacity { FadeAnimator {} }

            onStatusChanged: {
                if (status === Loader.Error) {
                    console.warn("Unable to load share plugin file:", shareMethodLoader.source)
                }
            }

            Item {
                id: placeholder

                width: parent.width
                height: Math.max(shareMethodsColumn.height,
                                 Theme.itemSizeHuge * 2,
                                 Theme.paddingLarge*2 + errorLabel.height)

                InfoLabel {
                    id: errorLabel

                    anchors.centerIn: parent
                    anchors.verticalCenterOffset: -Theme.paddingLarge
                    width: parent.width - Theme.horizontalPageMargin*2
                    visible: shareMethodLoader.status === Loader.Error
                             || (sharingMethodsModel.ready && sharingMethodsModel.count === 0)

                    text: shareMethodLoader.status === Loader.Error
                            //% "Unable to load sharing app"
                          ? qsTrId("sailfishshare-la-load_app_error")
                          : (sharingMethodsModel.filterByMultipleFileSupport
                               //: User is trying to share multiple files, but there are no apps that support this action
                               //% "No apps available for multi-file sharing"
                             ? qsTrId("sailfishshare-la-no_apps_available_for_multi_file_sharing")
                               //: User is trying to share a file, but there are no apps that support file sharing
                               //% "No apps available for file sharing"
                             : qsTrId("sailfishshare-la-no_apps_available_for_file_sharing"))
                }
            }
        }

        BusyIndicator {
            id: loadingIndicator

            anchors {
                top: dialogHeader.bottom
                horizontalCenter: parent.horizontalCenter
            }
            height: Theme.itemSizeLarge
            running: !delayLoadingIndicator.running
                     && !shareMethodsColumn.visible
                     && shareMethodLoader.status === Loader.Null
        }

        Timer {
            id: delayLoadingIndicator

            interval: 100
            running: !shareMethodsColumn.visible
                     && shareMethodLoader.status === Loader.Null
        }

        Column {
            id: shareMethodsColumn

            enabled: !root.waitingForFile

            anchors.top: dialogHeader.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            topPadding: Theme.paddingLarge
            bottomPadding: Theme.paddingLarge

            visible: sharingMethodsModel.ready
                     && sharingMethodsModel.count > 0
                     && (shareMethodLoader.status !== Loader.Ready
                         || shareMethodLoader.item.objectName === "baseSharePlugin")

            opacity: visible ? 1 : 0
            Behavior on opacity { FadeAnimator {} }

            Repeater {
                model: sharingMethodsModel

                delegate: ShareMethodItem {
                    property bool requiresFile: model.requiresFile

                    state: shareUIPath === root.selectedMethodSource ? root.status : ""

                    onClicked: {
                        root.selectedTransferMethodInfo = sharingMethodsModel.get(model.index)
                        root.selectedMethodSource = shareUIPath
                        if(requiresFile && shareAction.convertToFileAvailable) {
                            root.waitingForFile = true
                            root.requestFiles()
                        } else {
                            shareAction.selectedTransferMethodInfo = sharingMethodsModel.get(model.index)
                            shareMethodLoader.setSource(shareUIPath,
                                                        { "shareAction": shareAction })
                        }
                        content.scrollToTop()
                    }
                }
            }
        }
    }

    DBusInterface {
        id: settingsApp

        service: "com.jolla.settings"
        path: "/com/jolla/settings/ui"
        iface: "com.jolla.settings.ui"
    }
}
