/****************************************************************************************
** SPDX-FileCopyrightText: 2013 Jolla Ltd.
** SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import "private"

SilicaMouseArea {
    id: root

    property alias text: label.text
    property alias description: desc.text
    property alias _label: label

    property bool checked
    property bool automaticCheck: true
    property real leftMargin: Theme.horizontalPageMargin
    property real rightMargin: Theme.horizontalPageMargin
    property real _rightPadding
    property bool down: pressed && containsMouse && !DragFilter.canceled
    property bool busy

    // This is only used by ButtonGroup - if ButtonGroup is removed, this should be also:
    property int __silica_textswitch

    width: parent ? parent.width : Screen.width
    implicitHeight: Math.max(toggle.height, desc.y + desc.height)

    highlighted: down || pressTimer.running
    cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor

    CheckBox {
        id: toggle

        anchors {
            left: parent.left
            leftMargin: root.leftMargin
        }

        height: Theme.itemSizeSmall
        width: Theme.dp(52)
        textSwitch: true
        checked: root.checked
        enabled: root.enabled
        busy: root.busy
        highlighted: root.highlighted
    }

    Label {
        id: label

        width: parent.width - label.x - root.rightMargin - root._rightPadding
        opacity: root.enabled ? 1.0 : Theme.opacityLow
        anchors {
            verticalCenter: toggle.verticalCenter
            // center on the first line if there are multiple lines
            verticalCenterOffset: lineCount > 1 ? (lineCount-1)*height/lineCount/2 : 0
            left: toggle.right
            leftMargin: Theme.dp(52) - toggle.width
        }
        wrapMode: Text.Wrap
    }
    Label {
        id: desc

        width: label.width
        height: text.length ? (implicitHeight + Theme.paddingMedium) : 0
        opacity: root.enabled ? 1.0 : Theme.opacityLow
        anchors.top: label.bottom
        anchors.left: label.left
        wrapMode: Text.Wrap
        font.pixelSize: Theme.fontSizeExtraSmall
        color: highlighted ? root.palette.secondaryHighlightColor : root.palette.secondaryColor
    }
    Timer {
        id: pressTimer

        interval: Theme.minimumPressHighlightTime
    }

    onPressed: {
        root.DragFilter.begin(mouse.x, mouse.y)
        pressTimer.restart()
    }
    onCanceled: {
        root.DragFilter.end()
        pressTimer.stop()
    }
    onPreventStealingChanged: if (preventStealing) root.DragFilter.end()

    onClicked: {
        if (automaticCheck && !busy) {
            checked = !checked
        }
    }

    // for testing
    function _toggle() {
        return toggle
    }
}
