/**
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Nemo.FileManager 1.0
import Sailfish.FileManager 1.0
import Aurora.Controls 1.0

SilicaGridView {
    id: root

    property string initialPath
    property string description
    property bool gridLayout
    property bool selectionMode
    property bool isSelectOneItem
    property bool virtualFolder
    property bool searchPage
    property bool pickerPage
    property bool isHomePageMode
    property var extractor
    property bool extractMode: extractor ? extractor.extractMode : false
    property bool archiveMode
    property bool pasteMode: FileEngine.clipboardCount > 0
    property bool standardPathSelected
    property var standardPaths: []
    property alias __gridView: root

    readonly property bool isRemoving: toolBar._deletionProcessing

    signal deleteOneClicked(string path)
    signal filesDeleted()
    signal multiselectRequest()

    function clearSelectedFiles(args) {
        if (!args) {
            args = { force: false }
        }

        if (!root.selectionMode || args.force) {
            if (model.clearSelectedFiles != null) {
                model.clearSelectedFiles()
            }
        }

        d.resetSelectOneItemFlags(root)
    }

    function openThirdColumn(url, properties) {
        if (extractor != null) {
            extractor.openThirdColumn(url, properties)
        } else {
            return pageStack.animatorPush(url, properties)
        }
    }

    interactive: !isSelectOneItem
    bottomMargin: root.selectionMode ? toolBar.height : 0

    onIsSelectOneItemChanged: {
        if (!root.isSelectOneItem && popupMenu.opened) {
            root.closePopupMenu()
        }
    }

    onSelectionModeChanged: {
        if (selectionMode) isSelectOneItem = false

        FileManager.multiselectionInProgress = root.selectionMode
    }

    NumberAnimation {
        id: contentYAnimation

        target: root
        property: "contentY"
        from: root.contentY
        duration: 200
    }

    FileGridFooter {
        id: toolBar

        readonly property bool needMove: root.currentItem
                                         ? root.currentItem.y >= root.contentY + root.height - toolBar.height * 2
                                         : false

        readonly property bool needShow: root.selectionMode ||
                                         root.pasteMode ||
                                         root.extractMode

        property bool _needMoveRestore: needMove

        anchors.bottomMargin: needShow ? SafeZoneRect.insets.bottom : -toolBar.height

        _bottomPosition: true
        parentGridView: root
        buttonWidth: parent.width / 4
        standardPath: standardPaths.length > 0 && (standardPathSelected || StandardPaths.home === model.path)

        Behavior on anchors.bottomMargin {
            NumberAnimation {
                duration: contentYAnimation.duration
            }
        }

        onNeedShowChanged: {
            if (!root.currentItem || (!toolBar.needMove && !toolBar._needMoveRestore)) {
                return
            }

            if (toolBar.needShow) {
                contentYAnimation.to = root.contentY + toolBar.height
                contentYAnimation.start()
                toolBar._needMoveRestore = true
            } else {
                contentYAnimation.to = root.contentY - toolBar.height
                contentYAnimation.start()
                toolBar._needMoveRestore = false
            }
        }

        onPasteClicked: {
            FileEngine.pasteFiles(model.path, true, indexIfExists)
            clearSelectedFiles()
        }

        onCancelClicked: {
            if (pasteMode) {
                FileEngine.clearClipboard()
                clearSelectedFiles()

                if (root.searchPage && pageStack.currentPage.__forceHideSearchField !== undefined) {
                    pageStack.currentPage.__forceHideSearchField = false
                }
            } else if (extractMode) {
                extractor.archiveFile = ""
                extractor.extractMode = false
                clearSelectedFiles()
            }
        }

        onExtractClicked: {
            extractor.extractAllFiles(model.path)
            extractor.archiveFile = ""
            extractor.extractMode = false
            clearSelectedFiles()
        }

        onShareClicked: {
            shareAction.resources = model.selectedFilesToShare()
            shareAction.trigger()

            root.clearSelectedFiles()
        }

        onCopyClicked: {
            FileEngine.copyFiles(model.selectedFiles())

            d.resetSeletionMode()
        }

        onOpenMenuClicked: {
            openPopupMenu()
        }

        onDeleteClicked: {
            const files = model.selectedFiles()

            if (!root.isHomePageMode) {
                toolBar._deletionProcessing = true

                openThirdColumn(Qt.resolvedUrl("RemoveEntityDialog.qml"), {
                    filePaths: files,
                    acceptedCallback: function(dialog) {
                        FileEngine.deleteFiles(files, true)
                        dialog.displayResultNotice()
                        filesDeleted()

                        root.clearSelectedFiles({ force: true })
                    },
                    destructionCallback: function(dialog) {
                        toolBar._deletionProcessing = false

                        // were cleared in accepted callback
                        if (!dialog.wasAccepted) {
                            root.clearSelectedFiles()
                        }
                    }
                })
            } else {
                if (files.length === 1)
                    root.deleteOneClicked(files[0])
            }
        }
    }

    ShareAction {
        id: shareAction
    }

    // Width of image background in FileItem
    readonly property real cellBackgroundSize: Theme.iconSizeLauncher //Theme.iconSizeMedium + Theme.paddingLarge * 2

    readonly property real contentCellWidthForLargeScreen: cellBackgroundSize + Theme.paddingLarge * 3 // Column width in FileItem
                                                           + Theme.paddingSmall * 3 // horizontal padding of Row in FileItem

    readonly property real chunkWidth: {
        var columns = Math.floor(width / contentCellWidthForLargeScreen)
        return (width - columns * contentCellWidthForLargeScreen) / columns
    }
    readonly property real cellWidthForLargeScreen: contentCellWidthForLargeScreen + chunkWidth

    readonly property real cellHeightForLargeScreen: cellBackgroundSize
                                                     + Theme.paddingSmall * 9
                                                     + nameMetrics.height * 2 + sizeMetrics.height
    readonly property real cellHeightForSmallScreen: cellBackgroundSize + Theme.paddingSmall * 3
    property real cellWidthForSmallScreen

    cellHeight: gridLayout ? cellHeightForLargeScreen : cellHeightForSmallScreen
    cellWidth: gridLayout ? cellWidthForLargeScreen : cellWidthForSmallScreen

    currentIndex: -1
    displaced: Transition { NumberAnimation { properties: "x,y"; easing.type: Easing.InOutQuad; duration: 200 } }
    cacheBuffer: cellHeight * 16
    width: parent.width

    function openPopupMenu(selectedFile, parent) {
        const selectedFiles = model.selectedFiles()

        if (!selectedFiles.length) {
            console.warn("no selected files")
            return
        }

        info.file = selectedFiles[0]
        popupMenu.parent = parent || toolBar.itemAnchorPopup
        popupMenu.open()
    }

    function closePopupMenu() {
        popupMenu.close()
    }

    function displayInitialSelectedFiles() {
        if (!FileManager.multiselectInitialSelectedFiles.length ||
                !(root.model.populated === undefined ? true : root.model.populated)
                || !root.model.count
                || !root.selectionMode) {
            return
        }

        FileManager.multiselectInitialSelectedFiles.forEach(function(item) {
            if ((!item.selected && item.selected !== undefined) ||
                    root.model.selectedCount === 0) {
                root.model.toggleSelectedFile(item.index)
            }
        })

        root.positionViewAtIndex(FileManager.multiselectInitialSelectedFiles[0].index,
                                 GridView.Center)

        FileManager.multiselectInitialSelectedFiles = []
    }

    Connections {
        target: FileManager.findPage(root)

        onStatusChanged: {
            if (status === PageStatus.Active) {
                FileManager.multiselectionInProgress = root.selectionMode

                if (root.selectionMode ||
                        root.pasteMode ||
                        !root.model.selectedCount) {
                    return
                }

                root.clearSelectedFiles()
                FileManager.homePageSelectionMode = false
            } else if (status === PageStatus.Deactivating) {
                if (root.selectionMode) {
                    FileManager.multiselectionInProgress = false
                }
            }
        }
    }

    Connections {
        target: FileManager
        onSelectedFilesCleared: {
            FileManager.homePageSelectionMode = false

            root.clearSelectedFiles()
        }
    }

    Connections {
        target: model

        onCountChanged: root.displayInitialSelectedFiles()

        onSelectedCountChanged: {
            if ((root.selectionMode || root.isSelectOneItem) && model.selectedCount !== 0) {
                var files = model.selectedFiles()
                for (var i = 0; i < files.length; i++) {
                    if (standardPaths.indexOf(files[i]) !== -1) {
                        root.standardPathSelected = true
                        return
                    }
                }
                root.standardPathSelected = false
            }
        }
    }

    Rectangle {
        id: backgroundOverlay

        anchors.fill: parent
        visible: root.isSelectOneItem || root.selectionMode && toolBar._deletionProcessing
        color: Theme.rgba("black", Theme.opacityFaint)
        z: parent.z + 1

        MouseArea {
            anchors.fill: parent
            enabled: {
                if (root.selectionMode) {
                    return false
                }

                return root.isSelectOneItem && !toolBar._deletionProcessing
            }

            onClicked: FileManager.selectedFilesCleared()
            onPositionChanged: FileManager.selectedFilesCleared()
        }
    }

    FileInfo {
        id: info
    }

    PopupMenuMain {
        id: popupMenu

        onDeleteClicked: toolBar.deleteClicked()

        onShareClicked: toolBar.shareClicked()

        onCopyClicked: toolBar.copyClicked()

        onDetailsClicked: {
            openThirdColumn(Qt.resolvedUrl("DetailsPage.qml"), {
                                fileName: info.fileName,
                                mimeType: info.mimeType,
                                isDir: info.isDir,
                                modified: info.modified,
                                size: info.size,
                                path: info.absolutePath
                            })
        }

        onRenameClicked: {
            openThirdColumn(Qt.resolvedUrl("RenameDialog.qml"), {
                                oldPath: info.absolutePath,
                                //% "Rename"
                                acceptText: qsTrId("filemanager-he-rename")
                            })
        }

        onCutClicked: {
            FileEngine.cutFiles(model.selectedFiles())

            d.resetSeletionMode()
        }

        onExtractClicked: {
            extractor.archiveFile = info.absolutePath
            extractor.destinationPageName = "__archive_page"
            extractor.rejectedPageName = "__directory_page"
            if (needSelection)
                extractor.extractMode = true

            if (!needSelection) {
                extractor.extractAllFiles(model.path)
                extractor.archiveFile = ""
                clearSelectedFiles()
            }

            root.isSelectOneItem = false
        }

        onCompressClicked: {
            openThirdColumn(Qt.resolvedUrl("CreateArchiveDialog.qml"), {
                                extractorView: extractor,
                                files: model.selectedFiles(),
                                currentDirectoryPath: typeof model.path !== "undefined" && model.path.length > 0 ? model.path : ""
                            })
        }

        onLocationClicked: {
            if (!FileEngine.isSymLink(info.absolutePath)) {
                openDirectory(info.directoryPath)
            } else {
                if (openSymLink) {
                    openDirectory(info.directoryPath)
                } else {
                    //% "Following a symbolic link is prohibited"
                    errorNotification.show(qsTrId("filemanager-la-open_symlink_failed"))
                }
            }
            clearSelectedFiles()
        }

        onSelectModeClicked: {
            FileManager.multiselectInitialSelectedFiles = [ popupMenu.parent.modelData ]

            root.multiselectRequest()
        }

        onClosed: root.clearSelectedFiles()
        onOpenWithClicked: URIOpener.openUriWith(Qt.resolvedUrl(info.absolutePath))

        virtualFolder: root.virtualFolder
        searchPage: root.searchPage
        standardPath: root.standardPaths.length > 0 && root.standardPathSelected
        isSelectOneItem: root.isSelectOneItem
        hasFiles: model.count > 0
        isUserDir: FileEngine.isUserDir(model.path)
        singleItemSelected: model.selectedCount === 1
        isArchive: info.isArchive
        isDir: info.isDir
    }

    Behavior on contentY {
        enabled: !gridLayout
        NumberAnimation { duration: 20 }
    }

    FontMetrics {
        id: nameMetrics
        // This value should be taken from nameLabel in FileItem when the screen is large
        font.pixelSize: Theme.fontSizeExtraSmall
    }
    FontMetrics {
        id: sizeMetrics
        // This value should be taken from sizeLabel in FileItem when the screen is large
        font.pixelSize: Theme.fontSizeTiny
    }

    QtObject {
        id: d

        function resetSeletionMode() {
            if (root.selectionMode || root.searchPage) {
                const previousPage = pageStack.previousPage() || (function() {
                    return pageStack.currentPage
                })()

                const gridView = previousPage.fileGridView || (function() {
                    for (var i = 0; i < previousPage.children.length; ++i) {
                        const child = previousPage.children[i]

                        if (child.__gridView) {
                            return child
                        }
                    }

                    return null
                })()

                if (gridView) {
                    if (previousPage.__forceHideSearchField !== undefined) {
                        previousPage.__forceHideSearchField = true
                    }

                    d.resetSelectOneItemFlags(gridView)

                    pageStack.pop(null, PageStackAction.Immediate)
                }
            } else {
                d.resetSelectOneItemFlags(root)
            }
        }

        function resetSelectOneItemFlags(gridView) {
            gridView.isSelectOneItem = false

            FileManager.homePageSelectOneItem = false
        }

        function resetSearchPageFlag(gridView) {
            gridView.searchPage = false
        }
    }

    Component.onCompleted: root.displayInitialSelectedFiles()
}
