/*
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QDBusReply>
#include <QDebug>
#include <QString>
#include <QDBusInterface>
#include <QDBusConnection>
#include <QDBusPendingReply>
#include <QDBusPendingCall>
#include <QDBusPendingCallWatcher>

#include <functional>

namespace {

template <typename T, typename... Args>
T callWithResCheck(const QString &moduleName, QDBusInterface &iface, const QString &method, T defaultValue, Args... args)
{
    QDBusReply<T> reply = iface.call(method, std::forward<Args>(args)...);
    if (reply.isValid())
        return reply.value();
    qWarning() << moduleName << "error reply on method" << method << ":" << reply.error().message();
    return defaultValue;
}

template <typename T, typename... Args>
bool callAndStoreWithResCheck(const QString &moduleName, QDBusInterface &iface, const QString &method, T &storeResTo, Args... args)
{
    QDBusReply<T> reply = iface.call(method, std::forward<Args>(args)...);
    if (reply.isValid()) {
        storeResTo = reply.value();
        return true;
    }
    qWarning() << moduleName << "error reply on method" << method << ":" << reply.error().message();
    return false;
}

template <typename T, typename... Args>
T callWithGetError(QDBusInterface &iface, QString &error, const QString &method, T defaultValue, Args... args)
{
    error.clear();
    QDBusReply<T> reply = iface.call(method, std::forward<Args>(args)...);
    if (reply.isValid())
        return reply.value();
    error = reply.error().message();
    return defaultValue;
}

template <typename... Args>
QString voidCallWithGetError(QDBusInterface &iface, const QString &method, Args... args)
{
    QDBusReply<void> reply = iface.call(method, std::forward<Args>(args)...);
    return reply.isValid() ? QString() : reply.error().message();
}

template <typename... Args>
bool voidCallWithResCheck(const QString &moduleName, QDBusInterface &iface, const QString &method, Args... args)
{
    QDBusReply<void> reply = iface.call(method, std::forward<Args>(args)...);
    if (reply.isValid())
        return true;
    qWarning() << moduleName << "error reply on method" << method << ":" << reply.error().message();
    return false;
}

template <typename ResType, typename... Args>
void makeAsyncCall(const QString &moduleName,
                   QDBusInterface &iface,
                   QObject *parent,
                   std::function<void (const QDBusPendingReply<ResType> &)> callback,
                   const QString &method,
                   Args... args) {
    QDBusPendingCall pcall = iface.asyncCall(method, std::forward<Args>(args)...);
    QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(pcall, parent);
    QObject::connect(watcher,
                     &QDBusPendingCallWatcher::finished,
                     [callback, moduleName, method](QDBusPendingCallWatcher *call) {
                         QDBusPendingReply<ResType> reply = *call;
                         if (reply.isError() || !reply.isValid())
                             qWarning() << moduleName << "error reply on method "
                                        << method << ":" << reply.error().message();
                         if (callback)
                            callback(reply);
                         call->deleteLater();
                     });
}

} // anonymous namespace
