/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013-2019 Jolla Ltd
** SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0

BackgroundItem {
    id: viewItem
    property var menu
    property bool menuOpen: _menuItem != null && _menuItem._open
    property bool openMenuOnPressAndHold: true

    property Item _menuItem
    property bool _menuItemCreated
    property bool _connectPressAndHold: openMenuOnPressAndHold && menu !== null && menu !== undefined

    property Item __silica_remorse_item
    onMenuOpenChanged: {
        var view = viewItem.ListView.view || viewItem.GridView.view
        if (view && ('__silica_contextmenu_instance' in view)) {
            if (menuOpen) {
                view.__silica_contextmenu_instance = _menuItem
            } else if (viewItem.__silica_contextmenu_instance === _menuItem) {
                view.__silica_contextmenu_instance = null
            }
        }
    }

    function openMenu(properties) {
        if (menu == null) {
            return null
        }
        if (_menuItem == null) {
            _initMenuItem(properties)
        } else {
            for (var prop in properties) {
                if (prop in _menuItem) {
                    _menuItem[prop] = properties[prop]
                }
            }
        }
        if (_menuItem) {
            _menuItem.open(viewItem)
        }
        return _menuItem
    }

    function closeMenu() {
        if (_menuItem != null) {
            _menuItem.close()
        }
    }

    function _initMenuItem(properties) {
        if (_menuItem || (menu == null)) {
            return
        }
        var result
        if (menu.createObject !== undefined) {
            result = menu.createObject(viewItem, properties || {})
            _menuItemCreated = true
            result.closed.connect(function() { _menuItem.destroy() })
        } else {
            result = menu
            _menuItemCreated = false
            for (var prop in properties) {
                if (prop in result) {
                    result[prop] = properties[prop]
                }
            }
        }
        _menuItem = result
    }

    highlighted: down || menuOpen
    implicitHeight: menuOpen ? _menuItem.height + contentItem.height : contentItem.height
    contentHeight: Theme.itemSizeSmall
    _backgroundColor: Theme.rgba(palette.highlightBackgroundColor, _showPress && !menuOpen ? Theme.highlightBackgroundOpacity : 0)

    on_ConnectPressAndHoldChanged: {
        if (_connectPressAndHold) {
            viewItem.pressAndHold.connect(handlePressAndHold)
            viewItem.clicked.connect(handleClicked)
        } else {
            viewItem.pressAndHold.disconnect(handlePressAndHold)
            viewItem.clicked.disconnect(handleClicked)
        }
    }

    function handlePressAndHold() {
        if (down)
            openMenu()
    }

    function handleClicked(mouse) {
        if (mouse.button == Qt.RightButton && openMenuOnPressAndHold) {
            openMenu()
        }
    }

    onMenuChanged: {
        if (menu != null && _menuItem != null) {
            if (_menuItemCreated) {
                // delete the previously created context menu instance
                _menuItem.destroy()
            }
            _menuItem = null
        }
    }

    Component.onDestruction: {
        if (_menuItem != null) {
            _menuItem.close()
            _menuItem._parentDestroyed()
        }
    }
}
