/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013-2018 Jolla Ltd.
** SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

/*

With labelVisible: true (default)

  -------------------------
  |                       |
  |     textTopMargin     |
  |                       |
  | - - - - - - - - - - - |
  |                       |
  |                       |
  |      contentItem      |
  |                       |
  |                       |
  ------------------------- background rule
  |  Theme.paddingSmall   |
  | - - - - - - - - - - - |
  |                       |
  |       labelItem       |
  |                       |
  | - - - - - - - - - - - |
  |  Theme.paddingSmall   |
  -------------------------


With labelVisible: false

  -------------------------
  |                       |
  |     textTopMargin     |
  |                       |
  | - - - - - - - - - - - |
  |                       |
  |                       |
  |                       |
  |     contentItem       |
  |                       |
  |                       |
  |                       |
  ------------------------- background rule
  |  Theme.paddingSmall   |
  | - - - - - - - - - - - |
  |  Theme.paddingSmall   |
  -------------------------

*/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Aurora.Primer 1.0
import "Util.js" as Util

TextBaseItem {
    id: textBase

    property string label
    property color color: textBase.highlighted ? palette.highlightColor : palette.primaryColor

    property color cursorColor: palette.primaryColor
    property alias placeholderText: placeholderTextLabel.text
    property alias placeholderColor: placeholderTextLabel.color

    property string description

    // internal
    property alias placeholderAnimationEnabled: placeholderBehavior.enabled
    property bool softwareInputPanelEnabled: true
    property bool errorHighlight: false
    // margins indicate the area around filled or underlined editor area
    property real textMargin: Theme.horizontalPageMargin
    property real textLeftMargin: textMargin + (leftItemContainer.active ? leftItemContainer.width + Theme.paddingMedium : 0)
    property real textRightMargin: textMargin

    property real textTopMargin: Theme.paddingSmall
    // paddings indicate how much space there is around text to the filled or underlined area
    property real textTopPadding: _filled ? Theme.paddingMedium : 0
    property real textLeftPadding: _filled ? Theme.paddingMedium : 0
    // note: not applied to rightItem.
    // if it's interactive, it can make sense to cover all the area to the end of the underline
    property real textRightPadding: _filled ? Theme.paddingMedium : 0

    // TODO: Change to use "placeholderTextLabel.lineHeight once merge request #298 is merged.
    readonly property real textVerticalCenterOffset: _totalTopMargins + placeholderTextLabel.height / 2
    property int selectionMode: TextInput.SelectCharacters
    property alias font: placeholderTextLabel.font
    property int focusOutBehavior: FocusBehavior.ClearItemFocus
    property bool autoScrollEnabled: true
    property int backgroundStyle: TextEditor.UnderlineBackground

    property Component background: Rectangle {
        x: textLeftMargin
        anchors.top: contentContainer.bottom
        width: parent.width - x - textRightMargin
        height: Math.round(Theme.pixelRatio)
        color: Theme.rgba(textBase._colorWithError, Theme.opacityHigh)
        visible: backgroundStyle != TextEditor.NoBackground

        Rectangle {
            id: focusHighlight

            property real focusLevel

            color: textBase._colorWithError
            height: parent.height + (_editor.activeFocus
                                     ? Math.round(Theme.pixelRatio) + (palette.colorScheme === Theme.DarkOnLight ? 1 : 0)
                                     : 0)
            width: focusLevel * parent.width
            states: State {
                name: "focused"
                when: _editor.activeFocus
                PropertyChanges {
                    target: focusHighlight
                    focusLevel: 1.0
                }
            }
            transitions: Transition {
                to: "focused"
                NumberAnimation { property: "focusLevel"; duration: 150 }
            }
        }

        Rectangle {
            visible: textBase._filled
            anchors.bottom: parent.top
            width: parent.width
            height: contentContainer.height
            color: _editor.activeFocus ? Theme.highlightColor : Theme.primaryColor
            opacity: 0.1
        }
    }

    property alias leftItem: leftItemContainer.item
    property alias rightItem: rightItemContainer.item

    // TODO: Remove this wrongly-formulated property name once users have been migrated, and version incremented
    property alias enableSoftwareInputPanel: textBase.softwareInputPanelEnabled

    property bool _suppressPressAndHoldOnText
    property Item _backgroundItem
    readonly property bool _filled: backgroundStyle == TextEditor.FilledBackground
    property QtObject _feedbackEffect
    property var _appWindow: __silica_applicationwindow_instance
    property Item _flickable
    property alias _errorIcon: errorIcon

    property alias _flickableDirection: flickable.flickableDirection
    property rect _autoScrollCursorRect: Qt.rect(0, 0, 0, 0)
    property int _handleSize: Math.round(Theme.dp(10)) * 2 // ensure even number
    property int _handleActiveAreaMinSize: Math.round(Theme.dp(36)) * 2 // ensure even number

    property Item _scopeItem: textBase
    property alias editor: textBase._editor  //XXX Deprecated
    property alias menu: clipboardPopupMenu.children

    // JB#45671: Deprecate labelVisible
    property bool labelVisible: true
    property bool hideLabelOnEmptyField: true

    property Component labelComponent: defaultLabelComponent

    property Component defaultLabelComponent: Component {
        TextEditorLabel {
            width: parent.width
            editor: textBase
        }
    }

    property Item _labelItem
    readonly property color _colorWithError: textBase.errorHighlight ? palette.errorColor : color

    default property alias _children: contentContainer.data
    property alias contentItem: contentContainer
    property alias _placeholderTextLabel: placeholderTextLabel
    property bool focusOnClick: !readOnly
    property bool _singleLine
    readonly property bool _isEmpty: text.length === 0 && !_editor.inputMethodComposing
    readonly property real _bottomMargin: Theme.paddingSmall + (labelVisible
                                                                ? labelItemContainer.height + labelItemContainer.anchors.bottomMargin
                                                                : Theme.paddingSmall) + descriptionLabel.height
    function forceActiveFocus() { _editor.forceActiveFocus() }
    function cut() { _editor.cut() }
    function copy() { _editor.copy() }
    function paste() { _editor.paste() }
    function select(start, end) { _editor.select(start, end) }
    function selectAll() { _editor.selectAll() }
    function selectWord() { _editor.selectWord() }
    function deselect() { _editor.deselect() }
    function positionAt(mouseX, mouseY) {
        var translatedPos = mapToItem(_editor, mouseX, mouseY)
        return _editor.positionAt(translatedPos.x, translatedPos.y)
    }
    function positionToRectangle(position) {
        var rect = _editor.positionToRectangle(position)
        var translatedPos = mapFromItem(_editor, rect.x, rect.y)
        rect.x = translatedPos.x
        rect.y = translatedPos.y
        return rect
    }

    function _fixupScrollPosition() {
        scrollProxy.HorizontalAutoScroll.fixup()
        scrollProxy.VerticalAutoScroll.fixup()
        VerticalAutoScroll.fixup()
    }

    onHorizontalAlignmentChanged: {
        if (explicitHorizontalAlignment) {
            placeholderTextLabel.horizontalAlignment = horizontalAlignment
        }
    }
    onExplicitHorizontalAlignmentChanged: {
        if (explicitHorizontalAlignment) {
            placeholderTextLabel.horizontalAlignment = horizontalAlignment
        } else {
            placeholderTextLabel.horizontalAlignment = undefined
        }
    }

    function _updateBackground() {
        if (_backgroundItem) {
            _backgroundItem.destroy()
            _backgroundItem = null
        }
        if (!readOnly && background && background.status) {
            _backgroundItem = background.createObject(textBase)
            _backgroundItem.z = -1
        }
    }

    VerticalAutoScroll.keepVisible: activeFocus && autoScrollEnabled

    // If the TextArea/Field has an implicit height we may need to scroll an external flickable to
    // keep the cursor in view.
    VerticalAutoScroll.cursorRectangle: {
        if (!autoScrollEnabled || !activeFocus) {
            return undefined
        }
        var cursor = _editor.cursorRectangle
        var left = Math.max(0, contentContainer.x + _editor.x + cursor.x - (Theme.paddingLarge / 2))
        var right = Math.min(width, left + cursor.width + Theme.paddingLarge)
        var top = Math.max(0, contentContainer.y + _editor.y + cursor.y - (Theme.paddingLarge / 2))
        var bottom = Math.min(height, top + cursor.height + Theme.paddingLarge)

        return Qt.rect(left, top , right - left, bottom - top)
    }

    function _updateLabelItem() {
        if (_labelItem) {
            _labelItem.destroy()
            _labelItem = null
        }

        _labelItem = labelComponent.createObject(labelItemContainer)
    }

    signal clicked(variant mouse)
    signal pressAndHold(variant mouse)

    highlighted: _editor.activeFocus

    opacity: enabled ? 1.0 : Theme.opacityLow

    property int _totalTopMargins: textTopMargin + textTopPadding
    property int _totalLeftMargins: textLeftMargin + textLeftPadding
    property int _totalRightMargins: textRightMargin + textRightPadding
    property int _rightItemWidth: rightItemContainer.active ? rightItemContainer.width : 0
    property int _totalVerticalMargins: Theme.paddingMedium + _totalTopMargins
                                        + (labelVisible ? labelItemContainer.height : 0)
                                        + (descriptionLabel.text.length > 0 ? descriptionLabel.height : Theme.paddingSmall)

    implicitHeight: _editor.height + _totalVerticalMargins
    implicitWidth: parent ? parent.width : Screen.width

    _keyboardPalette: {
        if (palette.colorScheme !== Theme.colorScheme
                    || palette.highlightColor !== Theme.highlightColor) {
            return JSON.stringify({
                "colorScheme": palette.colorScheme,
                "highlightColor": palette.highlightColor.toString()
            })
        } else {
            return ""
        }
    }

    onBackgroundChanged: _updateBackground()
    onLabelComponentChanged: _updateLabelItem()
    Component.onCompleted: {
        if (!_backgroundItem) {
            _updateBackground()
        }
        if (!_labelItem) {
            _updateLabelItem()
        }

        // Avoid hard dependency to feedback - NOTE: Qt5Feedback doesn't support TextSelection effect
        _feedbackEffect = Qt.createQmlObject("import QtQuick 2.0; import QtFeedback 5.0; ThemeEffect { effect: ThemeEffect.PressWeak }",
                                             textBase, 'ThemeEffect')

        // calling ThemeEffect.supported initializes the feedback backend,
        // without the initialization here the first playback drops few frames
        if (_feedbackEffect && !_feedbackEffect.supported) {
            _feedbackEffect = null
        }
    }

    // This is the container item for the editor.  It is not the flickable because we want mouse
    // interaction to extend to the full bounds of the item but painting to be clipped so that
    // it doesn't exceed the margins or overlap with the label text.
    Item {
        id: contentContainer

        property alias contentX: flickable.contentX
        property alias contentY: flickable.contentY

        clip: flickable.interactive

        anchors {
            fill: parent
            leftMargin: textLeftMargin
            topMargin: textTopMargin
            rightMargin: textRightMargin + _rightItemWidth
            bottomMargin: textBase._bottomMargin
        }
    }

    Label {
        id: placeholderTextLabel

        text: textBase.label
        color: textBase.highlighted ? palette.secondaryHighlightColor : palette.secondaryColor

        opacity: textBase._isEmpty ? 1.0 : 0.0
        Behavior on opacity {
            id: placeholderBehavior
            FadeAnimation {}
        }
        truncationMode: TruncationMode.Fade
        anchors {
            left: parent.left; top: parent.top; right: parent.right
            leftMargin: _totalLeftMargins
            topMargin: _totalTopMargins
            rightMargin: _totalRightMargins + _rightItemWidth
        }
    }

    OpacityRampEffect {
        id: rampEffect
        offset: 1 - 1 / slope

        states: [
            State {
                name: "verticalOpacityRamp"
                when: flickable.verticalFlick
                PropertyChanges {
                    target: rampEffect
                    sourceItem: contentItem
                    transformOrigin: Item.Center
                    slope: 1 + 10 * textBase.width / Screen.width
                    direction: {
                        if (flickable.contentY === flickable.originY) {
                            return OpacityRamp.TopToBottom
                        } else if (flickable.contentHeight - flickable.height <= flickable.contentY) {
                            return OpacityRamp.BottomToTop
                        } else {
                            return OpacityRamp.BothEnds
                        }
                    }
                }
            },
            State {
                name: "horizontalOpacityRamp"
                when: flickable.horizontalFlick
                PropertyChanges {
                    target: rampEffect
                    sourceItem: contentItem
                    slope: 1 + 20 * textBase.width / Screen.width
                    direction: {
                        if (flickable.contentX === flickable.originX) {
                            return OpacityRamp.LeftToRight
                        } else if (flickable.contentWidth - flickable.width <= flickable.contentX) {
                            return OpacityRamp.RightToLeft
                        } else {
                            return OpacityRamp.BothSides
                        }
                    }
                }
            }
        ]
    }

    Flickable {
        id: flickable

        readonly property bool verticalFlick: _editor.height > (contentContainer.height - textTopPadding)
        readonly property bool horizontalFlick: _editor.width > (contentContainer.width - textLeftPadding - textRightPadding)

        anchors {
            fill: parent
            leftMargin: textLeftMargin
            topMargin: textTopMargin
            rightMargin: textRightMargin +  _rightItemWidth
        }

        pixelAligned: true
        contentHeight: scrollProxy.height + textBase._bottomMargin
        contentWidth: scrollProxy.width + Theme.paddingSmall
        interactive: verticalFlick || horizontalFlick
        boundsBehavior: Flickable.StopAtBounds

        onMovingChanged: {
            if (!moving && selectionItem.wasOpened) {
                selectionItem.openMenu()
                selectionItem.wasOpened = false
            }
            if (moving && clipboardPopupMenu.opened) {
                selectionItem.wasOpened = true
            }
        }

        Item {
            id: scrollProxy
            width: textBase._editor.width + textLeftPadding + textRightPadding
            height: textBase._editor.height + textTopPadding

            HorizontalAutoScroll.animated: false
            HorizontalAutoScroll.cursorRectangle: textBase._editor.activeFocus && autoScrollEnabled
                                                  ? textBase._editor.cursorRectangle
                                                  : undefined
            HorizontalAutoScroll.leftMargin: Math.max(0, Math.min(
                        Theme.paddingLarge + Theme.paddingSmall,
                        textBase._editor.cursorRectangle.x))
            HorizontalAutoScroll.rightMargin: Math.max(0, Math.min(
                        Theme.paddingLarge + Theme.paddingSmall,
                        width - textBase._editor.cursorRectangle.x - textBase._editor.cursorRectangle.width))
                    + Theme.paddingSmall
            VerticalAutoScroll.animated: false
            VerticalAutoScroll.cursorRectangle: textBase._editor.activeFocus && autoScrollEnabled
                                                ? textBase._editor.cursorRectangle
                                                : undefined
            VerticalAutoScroll.topMargin: Math.max(0, Math.min(
                        Theme.paddingLarge / 2,
                        textBase._editor.cursorRectangle.y))
            VerticalAutoScroll.bottomMargin: Math.max(0, Math.min(
                        Theme.paddingLarge / 2,
                        height - textBase._editor.cursorRectangle.y - textBase._editor.cursorRectangle.height))
                    + textBase._bottomMargin // The interactive area of the flickable extends to the bottom of the root item. This part of the margin ensures the cursor never ventures into that space.
        }
    }

    MouseArea {
        id: mouseArea

        property real initialMouseX
        property real initialMouseY
        property bool hasSelection: _editor !== null && _editor.selectedText != ""
        property bool cursorHit
        property bool cursorGrabbed
        property bool handleGrabbed
        property bool textSelected
        property Item selectionStartHandle
        property Item selectionEndHandle
        property Item aloneHandle
        property int cursorStepThreshold: Theme.itemSizeSmall / 2
        property real scaleFactor: 2
        property int scaleOffset: Theme.itemSizeSmall / 2
        property int scaleTopMargin: Theme.paddingLarge
        property int touchAreaSize: Theme.itemSizeExtraSmall
        property int moveThreshold: Theme.paddingMedium
        property int touchOffset
        property bool showAloneHandle
        property int clickCount
        property real prevClickMouseX
        property real prevClickMouseY

        property rect previousHandleRect
        property int hSelectedCharCount

        function positionAt(mouseX, mouseY) {
            var clippedX = Math.min(Math.max(parent.anchors.leftMargin, mouseX), parent.width + parent.anchors.leftMargin)
            var clippedY = Math.min(Math.max(parent.anchors.topMargin, mouseY), parent.height + parent.anchors.topMargin)
            var translatedPos = mapToItem(_editor, clippedX, clippedY)
            translatedPos.x = Math.max(0, Math.min(_editor.width - 1, translatedPos.x))
            translatedPos.y = Math.max(0, Math.min(_editor.height - 1 , translatedPos.y))
            return _editor.positionAt(translatedPos.x, translatedPos.y)
        }

        function aloneHandlePositionHit(position, mouseX, mouseY) {
            var rect = _editor.positionToRectangle(position)
            const centerX = rect.x + rect.width * 0.5
            rect.height += textBase._handleSize
            rect.width = textBase._handleSize
            rect.x = centerX - textBase._handleSize * 0.5
            const centerY = rect.y + rect.height * 0.5

            if (rect.width < textBase._handleActiveAreaMinSize) {
                rect.width = textBase._handleActiveAreaMinSize
                rect.x = centerX - textBase._handleActiveAreaMinSize * 0.5
            }

            if (rect.height < textBase._handleActiveAreaMinSize) {
                rect.height = textBase._handleActiveAreaMinSize
                rect.y = centerY - textBase._handleActiveAreaMinSize * 0.5
            }

            var translatedPos = mapToItem(_editor, mouseX, mouseY)
            return translatedPos.x >= rect.x
                    && translatedPos.x <= rect.x + rect.width
                    && translatedPos.y >= rect.y
                    && translatedPos.y <= rect.y + rect.height
        }

        function positionHit(position, mouseX, mouseY) {
            var rect = _editor.positionToRectangle(position)
            var translatedPos = mapToItem(_editor, mouseX, mouseY)
            return translatedPos.x > rect.x - touchAreaSize / 2
                    && translatedPos.x < rect.x + touchAreaSize / 2
                    && translatedPos.y > rect.y
                    && translatedPos.y < rect.y + Math.max(rect.height, touchAreaSize)
        }

        function moved(mouseX, mouseY) {
            return (Math.abs(initialMouseX - mouseX) > moveThreshold ||
                    Math.abs(initialMouseY - mouseY) > moveThreshold)
        }

        function updateTouchOffsetAndScaleOrigin(reset) {
            if (_appWindow !== undefined) {
                var cursorRect = _editor.cursorRectangle
                const item = typeof _appWindow._rotatingItem === "undefined" ? null : _appWindow._rotatingItem
                var translatedPos = mapToItem(item, mouseX, mouseY)
                var offset = Math.min(cursorRect.height / 2 + scaleOffset / scaleFactor,
                                      (translatedPos.y - scaleTopMargin) / scaleFactor - cursorRect.height / 2)
                if (reset || offset > touchOffset) {
                    touchOffset = offset
                }

                var cursorPos = _editor.mapToItem(item, cursorRect.x, cursorRect.y)

                var originX = mouseArea.mapToItem(item, mouseX, 0).x
                var originY = 0
                const scale = typeof _appWindow._contentScale === "undefined" ? null : _appWindow._contentScale
                if (reset) {
                    originY = (cursorPos.y < (scaleFactor - 1) * cursorRect.height + scaleOffset + scaleTopMargin)
                            ? (scaleFactor * cursorPos.y - scaleTopMargin) / (scaleFactor - 1)
                            : cursorPos.y + cursorRect.height + scaleOffset / (scaleFactor - 1)
                } else {
                    var mappedOrigin = _appWindow.contentItem.mapToItem(item,
                                                                        scale ? scale.origin.x : 0.0,
                                                                        scale ? scale.origin.y : 0.0)
                    var scaledCursorHeight = cursorRect.height * scaleFactor / (scaleFactor - 1)
                    const windowHeight = item ? item.height : _appWindow.height
                    if (cursorPos.y < scaleTopMargin) {
                        originY = Math.max(0, mappedOrigin.y - scaledCursorHeight)
                    } else if (cursorPos.y + scaleFactor * cursorRect.height + Theme.paddingMedium > windowHeight) {
                        originY = Math.min(windowHeight, mappedOrigin.y + scaledCursorHeight)
                    } else {
                        originY = mappedOrigin.y
                    }
                }

                if (scale && item) {
                    var mappedPos = item.mapToItem(_appWindow.contentItem, originX, originY)
                    scale.origin.x = mappedPos.x
                    scale.origin.y = mappedPos.y
                }
            }
        }

        function reset() {
            updateScale(false)
            selectionTimer.stop()
            cursorHit = false
            cursorGrabbed = false
            handleGrabbed = false
            preventStealing = false
            textSelected = false
        }

        function clicksFinished() {
            clickTimer.stop()

            if (clickCount < 2) {
                clickCount = 0
                return
            }

            var origSelectionStart = _editor.selectionStart
            var origSelectionEnd = _editor.selectionEnd

            if (clickCount === 2) {
                selectWord()
            } else if (clickCount === 3) {
                selectSentence()
            } else {
                selectParagraph()
            }

            if (origSelectionStart !== _editor.selectionStart || origSelectionEnd !== _editor.selectionEnd) {
                if (_feedbackEffect) {
                    _feedbackEffect.play()
                }
                mouseArea.textSelected = true
            }
            mouseArea.cursorHit = false

            clickCount = 0
        }

        function selectWord() {
            var translatedPos = mouseArea.mapToItem(_editor, prevClickMouseX, prevClickMouseY)

            Qt.inputMethod.commit()
            if (_editor.length === 0 || selectionTimer.positionAfter(_editor.length - 1, translatedPos.x, translatedPos.y)) {
                // This selection is outside the text itself - deselect and pass through as press-and-hold
                _editor.deselect()
                mouseArea.reset()
                return
            }

            _editor.cursorPosition = mouseArea.positionAt(prevClickMouseX, prevClickMouseY)
            selectionItem.select(_editor.selectWord)
        }

        function selectSentence() {
            Qt.inputMethod.commit()
            if (_editor.length === 0) {
                return
            }

            const cursorPosition = _editor.cursorPosition

            var start = _editor.text.lastIndexOf(".", cursorPosition - 1)
            start = Math.max(start, _editor.text.lastIndexOf("!", cursorPosition - 1))
            start = Math.max(start, _editor.text.lastIndexOf("?", cursorPosition - 1))
            start = Math.max(start, _editor.text.lastIndexOf(";", cursorPosition - 1))
            start = Math.max(start, _editor.text.lastIndexOf("\n", cursorPosition - 1))

            if (start === -1) {
                start = 0
            } else {
                start += 1
                while (start < _editor.text.length && /\s/.test(_editor.text[start])) {
                    ++start
                }
            }

            const dotNextPosition = _editor.text.indexOf(".", cursorPosition)
            const exclamationMarkNextPosition = _editor.text.indexOf("!", cursorPosition)
            const questionMarkNextPosition = _editor.text.indexOf("?", cursorPosition)
            const semicolonNextPosition = _editor.text.indexOf(";", cursorPosition)
            const newLineNextPosition = _editor.text.indexOf("\n", cursorPosition)

            var end = _editor.text.length

            if (dotNextPosition >= 0) {
                end = Math.min(end, dotNextPosition)
            }

            if (exclamationMarkNextPosition >= 0) {
                end = Math.min(end, exclamationMarkNextPosition + 1)
            }

            if (questionMarkNextPosition >= 0) {
                end = Math.min(end, questionMarkNextPosition + 1)
            }

            if (semicolonNextPosition >= 0) {
                end = Math.min(end, semicolonNextPosition + 1)
            }

            if (newLineNextPosition >= 0) {
                end = Math.min(end, newLineNextPosition)
            }

            selectionItem.select(function() { _editor.select(start, end) })
        }

        function selectParagraph() {
            Qt.inputMethod.commit()
            if (_editor.length === 0) {
                return
            }

            const cursorPosition = _editor.cursorPosition

            var start = cursorPosition
            while (start > 0 && _editor.text[start - 1] !== '\n') {
                start--
            }

            var end = cursorPosition
            while (end < _editor.text.length && _editor.text[end] !== '\n') {
                end++
            }

            selectionItem.select(function() { _editor.select(start, end) })
        }

        function updateScale(scaleEnabled) {
            if (_appWindow !== undefined && typeof _appWindow._contentScale !== "undefined") {
                if (scaleEnabled) {
                    _appWindow._contentScale.xScale = mouseArea.scaleFactor
                    _appWindow._contentScale.yScale = mouseArea.scaleFactor
                    mouseArea.updateTouchOffsetAndScaleOrigin(true)
                } else {
                    _appWindow._contentScale.xScale = 1.0
                    _appWindow._contentScale.yScale = 1.0
                }
            }
        }

        parent: flickable
        width: textBase.width
        height: textBase.height
        x: -parent.anchors.leftMargin
        y: -parent.anchors.topMargin
        enabled: textBase.enabled
        cursorShape: textBase.enabled ? Qt.IBeamCursor : Qt.ArrowCursor

        onPressed: {
            selectionItem.closeMenu()

            if (!_editor.activeFocus) {
                return
            }
            initialMouseX = mouseX
            initialMouseY = mouseY
            if (!hasSelection) {
                if (aloneHandlePositionHit(_editor.cursorPosition, mouseX, mouseY)) {
                    cursorHit = true
                    previousHandleRect = _editor.positionToRectangle(_editor.cursorPosition)
                }
            } else if (positionHit(_editor.selectionStart, mouseX, mouseY)) {
                handleGrabbed = true
                var selectionStart = _editor.selectionStart
                _editor.cursorPosition = _editor.selectionEnd
                _editor.moveCursorSelection(selectionStart, TextInput.SelectCharacters)
                preventStealing = true
                previousHandleRect = _editor.positionToRectangle(selectionStart)
            } else if (positionHit(_editor.selectionEnd, mouseX, mouseY)) {
                handleGrabbed = true
                var selectionEnd = _editor.selectionEnd
                _editor.cursorPosition = _editor.selectionStart
                _editor.moveCursorSelection(selectionEnd, TextInput.SelectCharacters)
                preventStealing = true
                previousHandleRect = _editor.positionToRectangle(selectionEnd)
            }

            if (!handleGrabbed) {
                selectionTimer.resetAndRestart()
            }
        }

        onClicked: {
            textBase.clicked(mouse)

            const touchDoubleTapDistance = Theme.dp(36)

            if (!clickTimer.running) {
                ++clickCount
                clickTimer.restart()
            } else if (Math.abs(mouseX - prevClickMouseX) <= touchDoubleTapDistance
                       && Math.abs(mouseY - prevClickMouseY) <= touchDoubleTapDistance) {
                ++clickCount
                if (clickCount >= 4) {
                    clicksFinished()
                } else {
                    clickTimer.restart()
                }
            } else {
                clickCount = 1
                clickTimer.restart()
            }

            prevClickMouseX = mouseX
            prevClickMouseY = mouseY
        }

        onPressAndHold: {
            if (_editor.activeFocus && _editor.text === "") {
                selectionItem.openMenu()
            }

            if (!_editor.activeFocus || !_suppressPressAndHoldOnText) {
                textBase.pressAndHold(mouse)
            }
        }

        onPositionChanged: {
            if (!handleGrabbed && !cursorGrabbed && moved(mouseX, mouseY)) {
                selectionTimer.stop()
                if (cursorHit) {
                    cursorGrabbed = true
                    preventStealing = true
                    if (aloneHandle === null) {
                        aloneHandle = aloneHandleComponent.createObject(_editor)
                    }
                    showAloneHandle = true
                    Qt.inputMethod.commit()
                }
            }
            if (handleGrabbed || cursorGrabbed) {
                if (_appWindow !== undefined && typeof _appWindow._contentScale !== "undefined"
                    && _appWindow._contentScale.animationRunning) {
                    // Don't change the cursor position during animation
                    return
                }
                updateTouchOffsetAndScaleOrigin(false)
                var cursorPosition = mouseArea.positionAt(mouseX, mouseY - mouseArea.touchOffset)
                if (handleGrabbed) {
                    _editor.moveCursorSelection(cursorPosition, textBase.selectionMode)
                } else {
                    _editor.cursorPosition = cursorPosition
                }
                const rect = _editor.positionToRectangle(cursorPosition)
                // Before scale was applied to any grab of any handle and in any way.
                // Now logic is: If we moving vertically - no need of scale at all. But if we move
                // at least 2 symbols in horizontal direction, this is temporary solution. But right
                // now it gives us time to find out better solution of this problem.
                if (previousHandleRect.y == rect.y) {
                    if (previousHandleRect.x != rect.x) {
                        // We on the same line but move horizontally
                        if (hSelectedCharCount > 1) {
                            updateScale(true)
                        } else {
                            hSelectedCharCount++
                        }
                    }
                } else {
                    hSelectedCharCount = 0
                    updateScale(false)
                }

                previousHandleRect = rect
            }
        }

        onReleased: {
            if (!handleGrabbed && !textSelected && !cursorGrabbed && containsMouse && (focusOnClick || _editor.activeFocus)) {
                Qt.inputMethod.commit()
                var translatedPos = mouseArea.mapToItem(_editor, mouseX, mouseY)
                var cursorRect = _editor.positionToRectangle(_editor.cursorPosition)
                var cursorPosition = _editor.cursorPosition

                // TODO: RTL text should mirror these. at RTL/LTR text block borders should avoid jumping cursor visually far away
                if (translatedPos.x < cursorRect.x && translatedPos.x > cursorRect.x - cursorStepThreshold &&
                        translatedPos.y > cursorRect.y && translatedPos.y < cursorRect.y + cursorRect.height) {
                    // step one character backward (unless at line start)
                    if (cursorPosition > 0 && (_editor.positionToRectangle(cursorPosition - 1).x < cursorRect.x)) {
                        cursorPosition = _editor.cursorPosition - 1
                    }
                } else if (translatedPos.x > cursorRect.x + cursorRect.width &&
                           translatedPos.x < cursorRect.x + cursorRect.width + cursorStepThreshold &&
                           translatedPos.y > cursorRect.y && translatedPos.y < cursorRect.y + cursorRect.height) {
                    // step one character forward
                    if (_editor.positionToRectangle(cursorPosition + 1).x > cursorRect.x) {
                        cursorPosition = _editor.cursorPosition + 1
                    }
                }

                if (cursorPosition === _editor.cursorPosition) {
                    cursorPosition = mouseArea.positionAt(mouseX, mouseY)
                    // NOTE: check for line change might fail, but currently don't care for such minor case
                    if (cursorPosition > 1 &&
                            _editor.positionToRectangle(cursorPosition - 1).y === _editor.positionToRectangle(cursorPosition).y &&
                            _editor.text.charAt(cursorPosition - 1) == ' ' &&
                            _editor.text.charAt(cursorPosition - 2) != ' ' &&
                            cursorPosition !== _editor.text.length) {
                        // space hit, move to the end of the previous word
                        cursorPosition--
                    }
                }
                _editor.cursorPosition = cursorPosition
                if (_editor.activeFocus) {
                    if (textBase.softwareInputPanelEnabled) {
                        Qt.inputMethod.show()
                    }
                } else {
                    _editor.forceActiveFocus()
                }
            }

            if (!cursorGrabbed && (cursorHit || handleGrabbed)) {
                if (!hasSelection && !showAloneHandle) {
                    if (aloneHandle === null) {
                        aloneHandle = aloneHandleComponent.createObject(_editor)
                    }
                    showAloneHandle = true
                } else {
                    selectionItem.openMenu()
                }
            }

            reset()
        }

        onCanceled: reset()

        onHasSelectionChanged: {
            showAloneHandle = false
            if (selectionStartHandle === null) {
                selectionStartHandle = handleComponent.createObject(_editor)
                selectionStartHandle.start = true
            }
            if (selectionEndHandle === null) {
                selectionEndHandle = handleComponent.createObject(_editor)
                selectionEndHandle.start = false
            }
        }

        Timer {
            id: clickTimer

            interval: 200
            onTriggered: mouseArea.clicksFinished()
        }
    }

    Item {
        id: selectionItem

        property bool wasOpened
        property bool selectedByTouch

        function select(method) {
            selectedByTouch = true
            method()
        }

        function openMenu() {
            selectedByTouch = false

            if (!_editor.focus) {
                return
            }

            const cursorRectStart = _editor.positionToRectangle(_editor.selectionStart)
            const cursorRectEnd = _editor.positionToRectangle(_editor.selectionEnd)
            const cursorPointStart = textBase.mapFromItem(_editor,
                                                          cursorRectStart.x,
                                                          cursorRectStart.y)
            const cursorPointEnd = textBase.mapFromItem(_editor,
                                                        cursorRectEnd.x,
                                                        cursorRectEnd.y)

            y = Math.min(parent.height - _editor.cursorRectangle.height,
                         Math.max(0, cursorPointStart.y - textBase._handleSize * 0.5))

            if (cursorPointStart.y === cursorPointEnd.y) {
                // The same line
                height = _editor.cursorRectangle.height + textBase._handleSize * 1.5
                x = cursorPointStart.x
                width = cursorPointEnd.x - cursorPointStart.x
            } else {
                height = Math.min(_editor.cursorRectangle.height + (cursorPointEnd.y - y) + textBase._handleSize * 1.5,
                                  parent.height - y)
                x = 0
                width = textBase.width
            }

            clipboardPopupMenu.open(selectionItem)
        }

        function closeMenu() {
            if (clipboardPopupMenu.opened) {
                clipboardPopupMenu.close()
            }
        }

        Connections {
            target: _editor

            onSelectedTextChanged: {
                selectionOpenTimer.restart()
            }

            onTextChanged: {
                mouseArea.showAloneHandle = false
                selectionItem.closeMenu()
            }
        }

        Timer {
            id: selectionOpenTimer

            interval: 1

            onTriggered: {
                if (_editor.selectedText && !mouseArea.handleGrabbed && !selectionTimer.running && selectionItem.selectedByTouch) {
                    selectionItem.openMenu()
                }
            }
        }

        ClipboardPopupMenu {
            id: clipboardPopupMenu

            palette: textBase.palette

            ClipboardPopupMenuItem {
                //% "Copy"
                text: qsTrId("components-la-copy")
                icon.source: "image://theme/icon-splus-copy-alt"
                visible: mouseArea.hasSelection

                onClicked: textBase.copy()
            }

            ClipboardPopupMenuItem {
                //% "Cut"
                text: qsTrId("components-la-cut")
                icon.source: "image://theme/icon-splus-cut"
                visible: mouseArea.hasSelection && !textBase._editor.readOnly

                onClicked: textBase.cut()
            }

            ClipboardPopupMenuItem {
                //% "Paste"
                text: qsTrId("components-la-paste")
                icon.source: "image://theme/icon-splus-paste"
                visible: textBase._editor.canPaste && !textBase._editor.readOnly

                onClicked: textBase.paste()
            }

            ClipboardPopupMenuItem {
                //% "Select all"
                text: qsTrId("components-la-select_all")
                icon.source: "image://theme/icon-splus-select-all"
                visible: textBase.selectedText !== textBase.text

                onClicked: selectionItem.select(_editor.selectAll)
            }

            contentItem.children: [
                InverseMouseArea {
                    anchors.fill: parent

                    enabled: clipboardPopupMenu.opened

                    onClickedOutside: closePopupTimer.start()
                    onPressedOutside: clipboardPopupMenu.close()
                }
            ]
        }

        // This helps stay in sync popup's inverse area and textBase inverse area
        Timer {
            id: closePopupTimer
            interval: 1
        }
    }

    InverseMouseArea {
        anchors.fill: parent
        enabled: _editor.activeFocus && textBase.softwareInputPanelEnabled
        onClickedOutside: {
            // If popup is open - so make sure click wasn't on popup, otherwise we loose focus
            if (closePopupTimer.running || !clipboardPopupMenu.opened) {
                focusLossTimer.start()
            }
        }
    }

    Item {
        id: labelItemContainer

        anchors {
            left: parent.left; bottom: descriptionLabel.top; right: parent.right
            leftMargin: _totalLeftMargins
            rightMargin: _totalRightMargins
            bottomMargin: {
                if (descriptionLabel.text.length > 0) {
                    return 0
                } else {
                    return readOnly ? Theme.paddingMedium : Theme.paddingSmall
                }
            }
        }
        visible: labelVisible
        height: children.length > 0 ? children[0].height : 0
    }

    onDescriptionChanged: if (description.length > 0) descriptionLabel.text = description

    Label {
        id: descriptionLabel

        text: description
        wrapMode: Text.Wrap

        color: textBase.errorHighlight ? palette.errorColor
                                       : placeholderTextLabel.color

        font.pixelSize: Theme.fontSizeExtraSmall

        opacity: description.length > 0 ? 1.0 : 0.0
        Behavior on opacity { FadeAnimator {}}
        height: description.length > 0 ? implicitHeight : 0
        Behavior on height { NumberAnimation { duration: 200; easing.type: Easing.InOutQuad } }

        bottomPadding: readOnly ? Theme.paddingMedium : Theme.paddingSmall
        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
            leftMargin: _totalLeftMargins
            rightMargin: textRightMargin
            bottomMargin: description.length > 0 && (textBase._isEmpty && textBase.hideLabelOnEmptyField) ? labelItemContainer.height
                                                                      : 0
        }
        Behavior on anchors.bottomMargin { NumberAnimation { duration: 200; easing.type: Easing.InOutQuad } }
    }

    Timer {
        id: selectionTimer

        property int counter

        repeat: true

        function resetAndRestart() {
            counter = 0
            interval = 800
            restart()
        }

        function positionAfter(position, mouseX, mouseY) {
            var rect = _editor.positionToRectangle(position)
            return mouseY > rect.y + Math.max(rect.height, mouseArea.touchAreaSize) ||
                    (mouseY >= rect.y &&
                     mouseX > rect.x + rect.width + mouseArea.touchAreaSize / 2)
        }

        onRunningChanged: {
            if (!running && _editor.selectedText && !mouseArea.handleGrabbed && selectionTimer.counter) {
                selectionItem.openMenu()
            }
        }

        onTriggered: {
            var origSelectionStart = _editor.selectionStart
            var origSelectionEnd = _editor.selectionEnd
            var translatedPos = mouseArea.mapToItem(_editor, mouseArea.initialMouseX, mouseArea.initialMouseY)
            if (counter == 0) {
                if (_suppressPressAndHoldOnText) {
                    Qt.inputMethod.commit()
                    if (_editor.length == 0 || positionAfter(_editor.length - 1, translatedPos.x, translatedPos.y)) {
                        // This selection is outside the text itself - deselect and pass through as press-and-hold
                        _editor.deselect()
                        mouseArea.reset()
                        textBase.pressAndHold({ 'x': mouseArea.initialMouseX, 'y': mouseArea.initialMouseY })
                        return
                    }
                }

                _editor.cursorPosition = mouseArea.positionAt(mouseArea.initialMouseX, mouseArea.initialMouseY)
                selectionItem.select(_editor.selectWord)
                interval = 600
                // single line editor to skip choosing visible area
                if (textBase._singleLine) {
                    counter++
                }
            } else if (counter == 1) {
                selectionItem.select(function() {
                    _editor.select(mouseArea.positionAt(0, translatedPos.y),
                                   mouseArea.positionAt(_editor.width, translatedPos.y))
                })
            } else {
                _editor.cursorPosition = _editor.text.length
                selectionItem.select(_editor.selectAll)
                stop()
            }
            if (origSelectionStart != _editor.selectionStart || origSelectionEnd != _editor.selectionEnd) {
                if (_feedbackEffect) {
                    _feedbackEffect.play()
                }
                mouseArea.textSelected = true
            }
            mouseArea.cursorHit = false
            counter++
        }
    }

    Timer {
        id: focusLossTimer
        interval: 1
        onTriggered: {
            selectionItem.closeMenu()

            // Note: textBase.focus.  Removing focus from the editor item breaks the focus
            // inheritence chain making it impossible for the editor to regain focus without
            // using forceActiveFocus()
            if (!textBase.activeFocus) {
            } else if (textBase.focusOutBehavior === FocusBehavior.ClearItemFocus) {
                textBase.focus = false
            } else if (textBase.focusOutBehavior === FocusBehavior.ClearPageFocus) {
                // Just remove the focus from the application window (that is a focus scope).
                // This allows an item to clear its active focus without breaking the focus
                // chain within a page.
                if (_appWindow !== undefined) {
                    _appWindow.focus = false
                } else {
                    textBase.focus = false // fallback
                }
            } else if (!_editor.activeFocus) {
                // Happens e.g. when keyboard is closed
                textBase.focus = false
            } else {
                _editor.deselect()
            }
            _editor.focus = true
            mouseArea.showAloneHandle = false
        }
    }

    Icon {
        id: errorIcon
        parent: null
        color: Theme.errorColor
        highlightColor: Theme.errorColor
        source: textBase.errorHighlight ? "image://theme/icon-splus-error" : ""
    }

    FontMetrics {
        id: fontMetrics
        font: _editor.font
    }

    TextBaseExtensionContainer {
        id: leftItemContainer

        x: textBase.textLeftMargin - (leftItemContainer.active ? leftItemContainer.width + Theme.paddingMedium : 0)
        y: (fontMetrics.height + _totalVerticalMargins - height)/2
        parent: textBase
    }

    TextBaseExtensionContainer {
        id: rightItemContainer

        y: ((_editor.y - textTopPadding) + fontMetrics.height)/2 - height/2 + _totalTopMargins
        parent: textBase
        anchors {
            right: parent.right
            rightMargin: textBase.textRightMargin
        }

        item: errorHighlight ? errorIcon : null
    }

    Connections {
        ignoreUnknownSignals: true
        target: _editor
        onActiveFocusChanged: {
            if (_editor.activeFocus) {
                if (textBase.softwareInputPanelEnabled) {
                    Qt.inputMethod.show()
                }
            } else {
                // When keyboard is explicitly closed (by swipe down) only _editor.focus is cleared.
                // Need to use focusLossTimer for clearing the focus of the parent.
                // (See the comments in focusLossTimer.)
                focusLossTimer.start()
            }
        }
    }
    Component {
        id: aloneHandleComponent

        Rectangle {
            property rect cursorRect: _editor.cursorRectangle

            color: palette.highlightBackgroundColor
            border.color: palette.secondaryColor
            border.width: Theme._lineWidth
            parent: _editor
            x: Math.round(cursorRect.x + cursorRect.width * 0.5 - width * 0.5)
            y: Math.round(cursorRect.y + cursorRect.height)
            width: textBase._handleSize
            height: width
            radius: width * 0.5
            smooth: true
            // Enable animation only when appearing
            opacity: visible ? 1 : 0
            visible: !mouseArea.hasSelection && mouseArea.showAloneHandle && textBase.activeFocus

            Behavior on opacity { FadeAnimation { duration: 100 } }
        }
    }

    Component {
        id: handleComponent

        Rectangle {
            id: handleId
            property bool start
            property var cursorRect: {
                _editor.width // creates a binding. we want to refresh the cursor rect e.g. on orientation change
                _editor.positionToRectangle(start ? _editor.selectionStart : _editor.selectionEnd)
            }

            color: palette.highlightBackgroundColor
            border.color: palette.secondaryColor
            border.width: Theme.dp(1)
            parent: _editor
            x: Math.round(cursorRect.x + cursorRect.width / 2 - width / 2)
            y: start ? Math.round(cursorRect.y - height / 2)
                     : Math.round(cursorRect.y + cursorRect.height - height / 2)
            width: textBase._handleSize
            height: width
            radius: width / 2
            smooth: true
            // Enable animation only when appearing
            visible: mouseArea.hasSelection && textBase.activeFocus
            opacity: visible ? 1 : 0

            Behavior on opacity { FadeAnimation { duration: 100 } }
        }
    }
}
