/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2022-2024 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: BSD 3-Clause
**
****************************************************************************************/

import QtQuick 2.3
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private

Item {
    id: root

    default property alias children: content.data

    Item {
        id: content

        readonly property var positionBinding: {
            var horisontal = 0
            var vertical = 0
            var docRoot = root
            do {
                vertical += docRoot.y
                horisontal += docRoot.x
                docRoot = docRoot.parent
            } while (docRoot)

            return {
                'horisontal' : horisontal,
                'vertical' : vertical,
                'insets' : SafeZoneRect.insets
            }
        }

        function update() {
            const windowOrientation = __silica_applicationwindow_instance.orientation
            var rect = root.mapToItem(__silica_applicationwindow_instance,
                                      root.x, root.y,
                                      root.width, root.height)

            var intersects = true

            if (__silica_applicationwindow_instance.displayMode === ApplicationDisplayMode.FillScreen
                    && (windowOrientation & Orientation.LandscapeMask)) {
                const rectMinX = rect.x
                const rectMaxX = rect.x + rect.width
                const cutoutMinX = Private.CutoutArea.cutoutRect.x
                const cutoutMaxX = Private.CutoutArea.cutoutRect.x + Private.CutoutArea.cutoutRect.width
                intersects = !(rectMinX > cutoutMaxX || rectMaxX < cutoutMinX)
            }

            if (intersects) {
                const windowWidth = __silica_applicationwindow_instance.width
                const windowHeight = __silica_applicationwindow_instance.height

                var safeZoneMinY = 0
                var safeZoneMinX = 0
                var safeZoneMaxX = 0
                var safeZoneMaxY = 0

                switch (windowOrientation) {
                case Orientation.Portrait:
                    safeZoneMinX = SafeZoneRect.insets.left
                    safeZoneMinY = SafeZoneRect.insets.top
                    safeZoneMaxX = windowWidth - SafeZoneRect.insets.right
                    safeZoneMaxY = windowHeight - SafeZoneRect.insets.bottom
                    break
                case Orientation.Landscape:
                    safeZoneMinX = SafeZoneRect.insets.bottom
                    safeZoneMinY = SafeZoneRect.insets.left
                    safeZoneMaxX = windowWidth - SafeZoneRect.insets.top
                    safeZoneMaxY = windowHeight - SafeZoneRect.insets.right
                    break
                case Orientation.PortraitInverted:
                    safeZoneMinX = SafeZoneRect.insets.right
                    safeZoneMinY = SafeZoneRect.insets.bottom
                    safeZoneMaxX = windowWidth - SafeZoneRect.insets.left
                    safeZoneMaxY = windowHeight - SafeZoneRect.insets.top
                    break
                case Orientation.LandscapeInverted:
                    safeZoneMinX = SafeZoneRect.insets.top
                    safeZoneMinY = SafeZoneRect.insets.right
                    safeZoneMaxX = windowWidth - SafeZoneRect.insets.bottom
                    safeZoneMaxY = windowHeight - SafeZoneRect.insets.left
                    break
                default:
                    break
                }

                if (rect.x < safeZoneMinX) {
                    rect.x = safeZoneMinX
                } else if (rect.x + rect.width > safeZoneMaxX) {
                    rect.x = safeZoneMaxX - rect.width
                }

                if (rect.y < safeZoneMinY) {
                    rect.y = safeZoneMinY
                } else if (rect.y + rect.height > safeZoneMaxY) {
                    rect.y = safeZoneMaxY - rect.height
                }
            }
            rect = root.mapFromItem(__silica_applicationwindow_instance,
                                    rect.x, rect.y,
                                    rect.width, rect.height)

            content.width = rect.width
            content.height = rect.height
            content.x = rect.x
            content.y = rect.y
        }

        onPositionBindingChanged: update()

        Connections {
            target: __silica_applicationwindow_instance
            onDisplayModeChanged: update()
            onOrientationChanged: update()
        }

        Component.onCompleted: update()
    }
}
