/*
 * SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import ru.auroraos.WebView.Private 1.0

TouchInput {
    id: touchInput

    signal pressed(var touchPoints)
    signal released(var touchPoints)
    signal touchUpdated(var touchPoints)
    signal canceled(var touchPoints)

    anchors.fill: parent

    MultiPointTouchArea {
        x: 0
        y: 0
        anchors.fill: parent
        maximumTouchPoints: touchInput.maximumTouchPoints
        enabled: touchInput.enabled
        mouseEnabled: true

        onPressed: {
            var touchPoints = filterOutsideTouchPoints(touchPoints)

            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchPressed(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.pressed(touchPoints)
        }

        // Release might happen outside of the MultiPointTouchArea, so we must not filter the points
        // Otherwise, CEF will continue thinking that a touch is active
        onReleased: {
            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchReleased(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.released(touchPoints)
        }

        // Cancel might happen outside of the MultiPointTouchArea, so we must not filter the points
        // Otherwise, CEF will continue thinking that a touch is active
        onCanceled: {
            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchCanceled(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.canceled(touchPoints)
        }

        onTouchUpdated: {
            var touchPoints = filterOutsideTouchPoints(touchPoints)

            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchMoved(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.touchUpdated(touchPoints)
        }

        onGestureStarted: {
            var points = filterOutsideTouchPoints(gesture.touchPoints)

            // Grab the gesture only when ALL touch points are inside
            if (points.length == gesture.touchPoints.length) {
                gesture.grab() // We grab the gesture to prevent an overlap with system gestures like
                               // interactions with a page stack
            }
        }

        // There is a bug in Qt 5.6 that makes the MultiPointTouchArea register events outside
        // of it when it is focused.
        //
        // In order to allow proper gesture grabbing outside of WebView and event passing
        // to the engine, we need to manually check the coords of each touch point.
        //
        // returns points that are inside the MultiPointTouchArea
        function filterOutsideTouchPoints(points) {
            var filteredPoints = []

            for (var i = 0; i < points.length; i++) {
                if (points[i].x < x || points[i].x > x + width ||
                    points[i].y < y || points[i].y > y + height) {
                    continue
                }
                filteredPoints.push(points[i])
            }

            return filteredPoints
        }
    }
}
