/**
 * SPDX-FileCopyrightText: 2020–2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0

Dialog {
    id: dialog
    objectName: "RenameDialog"

    property alias oldPath: fileInfo.file
    property alias nameField: fileName
    property alias archiveSettingsLoader: archiveSettings
    property alias flickable: dialogFlickable
    property alias acceptText: dialogHeader.acceptText

    property string directoryPath: fileInfo.directoryPath
    property string archiveFileName
    property bool archiveCreationMode
    property bool passwordDialogMode

    readonly property string _newPath: fileName.text !== "" && !_hasInvalidCharacters
                                       ? dialog.directoryPath + "/" + fileName.text
                                       : ""

    readonly property var _regExp: new RegExp("[\/*\?<>\|]+")
    readonly property var _regExpExtension: new RegExp(/\.[0-9a-z]+$/i)
    readonly property bool _hasInvalidCharacters: _regExp.test(fileName.text)
    readonly property bool _exists: {
        if (!archiveCreationMode) {
            if (fileInfo.isDir || !FileEngine.isUserDir(_newPath)) {
                return _newPath !== "" && _newPath !== oldPath && FileEngine.exists(_newPath)
            } else {
                return _newPath !== "" && _newPath === oldPath
            }
        } else {
            return FileEngine.exists(_newPath) && !FileEngine.isUserDir(_newPath)
        }
    }

    readonly property bool _hasFirstSpace: _newPath !== "" && /^\s+/.test(fileName.text)
    readonly property string fileNameWithoutExt: fileName.text.replace(_suffixForFileName(fileName.text), "")
    readonly property bool _hasFirstDot: _newPath !== "" && !fileInfo.isDir
                                         ? fileNameWithoutExt.charAt(0) === "." && fileNameWithoutExt.length === 1
                                         : fileName.text.charAt(0) === "." && fileName.length === 1
    readonly property bool _hasEmptyName: {
        var value = !fileInfo.isDir
        var matches = fileName.text.match(_regExpExtension)

        if (_newPath !== "" && (matches || []).length > 0)
            value = matches[0].length === fileName.text.length && !fileInfo.isDir

        return value
    }
    readonly property bool _tooLongName: encodeURI(fileName.text).split(/%..|./).length - 1 > 255

    function _suffixForFileName(fileName) {
        var suffix = FileEngine.extensionForFileName(fileName)
        return suffix !== "" ?  "." + suffix : suffix
    }

    FileInfo {
        id: fileInfo
    }

    SilicaFlickable {
        id: dialogFlickable

        anchors.fill: parent
        contentHeight: archiveItem.height + dialogHeader.height

        DialogHeader {
            id: dialogHeader
            objectName: "RenameDialogHeader"
        }

        Column {
            id: archiveItem

            anchors.top: dialogHeader.bottom
            width: parent.width
            spacing: Theme.paddingLarge

            DetailsIcon {
                objectName: "RenameDialogArchiveIcon"

                anchors.horizontalCenter: parent.horizontalCenter
                backgroundWidth: Theme.iconSizeExtraLarge + Theme.iconSizeExtraSmall
                backgroundHeight: backgroundWidth
                backgroundRadius: Theme.paddingMedium
                isDir: fileInfo.isDir
                mimeType: fileInfo.mimeType
                path: fileInfo.absolutePath
                iconSize: Theme.iconSizeExtraLarge
            }

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width - 2 * Theme.horizontalPageMargin
                visible: dialog.passwordDialogMode
                horizontalAlignment: Qt.AlignHCenter
                text: dialog.archiveFileName
            }


            TextField {
                id: fileName

                objectName: "FileNameTextField"

                width: parent.width
                visible: !passwordDialogMode
                enabled: visible
                label: {
                    if (dialog._hasInvalidCharacters || dialog._hasFirstDot) {
                        return fileInfo.isDir
                                ? qsTrId("filemanager-te-invalid_foldername")
                                  //% "Invalid file name"
                                : qsTrId("filemanager-te-invalid_filename")
                    } else if (dialog._exists) {
                        return fileInfo.isDir
                                ? qsTrId("filemanager-te-foldername_exists")
                                  //% "A file with the same name exists"
                                : qsTrId("filemanager-te-filename_exists")
                    } else if (dialog._hasFirstSpace) {
                        return fileInfo.isDir
                                ? qsTrId("filemanager-te-foldername_can`t_start_with_spaces")
                                  //% "File name can`t start with spaces"
                                : qsTrId("filemanager-te-filename_can`t_start_with_spaces")
                    } else if (dialog._tooLongName) {
                        return fileInfo.isDir
                                ? qsTrId("filemanager-te-foldername_too_long")
                                  //% "File name is too long"
                                : qsTrId("filemanager-te-filename_too_long")
                    } else {
                        //% "Title"
                        return qsTrId("filemanager-la-title")
                    }
                }

                placeholderText: qsTrId("filemanager-la-title")
                onFocusChanged: {
                    if (focus) {
                        var suffix = _suffixForFileName(text)

                        if (suffix.length > 0) {
                            select(0, text.length - suffix.length)
                        } else {
                            if (!fileInfo.isDir) {
                                cursorPosition = 0
                            } else {
                                select(0, text.length)
                            }
                        }
                    }
                }

                text: fileInfo.fileName
                errorHighlight: dialog._hasInvalidCharacters || dialog._exists || dialog._hasFirstSpace || dialog._hasFirstDot || dialog._tooLongName

                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.enabled: text !== ""
                EnterKey.onClicked: dialog.accept()

                Component.onCompleted: if (!dialog.passwordDialogMode) focus = true
            }

            Loader {
                id: archiveSettings

                width: parent.width
            }
        }

        VerticalScrollDecorator {}
    }

    canAccept: _newPath !== "" && !_exists && !_hasFirstSpace && !_hasEmptyName && !_hasFirstDot && !_tooLongName
    onAccepted: {
        if (_newPath !== oldPath && !archiveCreationMode) {
            FileEngine.rename(oldPath, _newPath)
        }
    }
}
