/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import QtQuick.Layouts 1.1
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0 as Nemo

Dialog {
    id: root

    property string text: {
        if (root.filePaths.length > 1) {
            //% "Are you sure you want to delete selected items (%n)?"
            return qsTrId("filemanager-la-delete_items_dialog_text", root.filePaths.length)
        }

        //% "Are you sure you want to delete this"
        const prefix = qsTrId("filemanager-la-delete_item_dialog_text_file_path")

        return "%1 %2 %3?"
               .arg(prefix)
               .arg(fileInfo.typeStringText)
               .arg(fileInfo.fileName)
    }

    property string hint: {
        if (root.filePaths.length > 1) {
            //% "Selected items will be deleted permanently."
            return qsTrId("filemanager-la-delete_items_dialog_hint")
        }

        var prefix = null
        var postfix = null

        if (fileInfo.isDir) {
            //% "Selected"
            prefix = qsTrId("filemanager-la-delete_item_dialog_hint_prefix_folder")

            //% "will be deleted permanently."
            postfix = qsTrId("filemanager-la-delete_item_dialog_hint_postfix_folder")
        } else {
            //% "Selected"
            prefix = qsTrId("filemanager-la-delete_item_dialog_hint_prefix_file")

            //% "will be deleted permanently."
            postfix = qsTrId("filemanager-la-delete_item_dialog_hint_postfix_file")
        }

        return "%1 %2 %3".arg(prefix).arg(fileInfo.typeStringHint).arg(postfix)
    }

    property var filePaths: []

    property var acceptedCallback: null
    property var rejectedCallback: null
    property var destructionCallback: null

    readonly property bool wasAccepted: d.wasAccepted

    function displayResultNotice() {
        if (root.filePaths.length > 1) {
            //% "Items have been deleted"
            Notices.show(qsTrId("filemanager-la-items_deleted"),
                         Notice.Short,
                         Notice.Bottom)
        } else {
            Notices.show("%1 ".arg(fileInfo.typeStringFirstLetterUpper) +
                         (fileInfo.isDir
                            //% "has been deleted"
                          ? qsTrId("filemanager-la-item_deleted_file_folder")
                            //% "has been deleted"
                          : qsTrId("filemanager-la-item_deleted_file_file")),
                         Notice.Short,
                         Notice.Bottom)
        }
    }

    canAccept: root.filePaths.length

    onAccepted: {
        if (root.acceptedCallback) {
            d.wasAccepted = true

            root.acceptedCallback(root)
        }
    }

    onRejected: {
        if (root.rejectedCallback) {
            d.wasAccepted = false

            root.rejectedCallback(root)
        }
    }

    Nemo.FileInfo {
        id: fileInfo

        readonly property string typeStringText: fileInfo.isDir
                                                   //% "folder"
                                                 ? qsTrId("filemanager-la-delete_item_dialog_folder_text")
                                                   //% "file"
                                                 : qsTrId("filemanager-la-delete_item_dialog_file")

        readonly property string typeStringHint: fileInfo.isDir
                                                   //% "folder"
                                                 ? qsTrId("filemanager-la-delete_item_dialog_folder_hint")
                                                   //% "file"
                                                 : qsTrId("filemanager-la-delete_item_dialog_file")

        readonly property string typeStringFirstLetterUpper:
            typeStringHint[0].toUpperCase() + typeStringHint.substring(1, typeStringHint.length)

        file: {
            if (!root.filePaths.length) {
                return ""
            }

            return root.filePaths[0]
        }
    }

    DialogHeader {
        id: header

        acceptText: qsTrId("filemanager-me-delete")
    }

    ColumnLayout {
        anchors {
            left: parent.left
            right: parent.right
            top: header.bottom
            leftMargin: Theme.horizontalPageMargin
            rightMargin: Theme.horizontalPageMargin
        }

        spacing: Theme.paddingLarge

        Label {
            Layout.fillWidth: true

            font.pixelSize: Theme.fontSizeExtraLarge
            wrapMode: Text.Wrap
            maximumLineCount: 5
            text: root.text
        }

        Label {
            Layout.fillWidth: true

            wrapMode: Text.Wrap
            text: root.hint

            color: Theme.rgba(Theme.primaryColor, Theme.opacityOverlay)
        }
    }

    QtObject {
        id: d

        property bool wasAccepted: false
    }

    Component.onDestruction: root.destructionCallback(root)
}
