/**
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.2
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Aurora.Controls 1.0

PopupMenu {
    id: root

    readonly property int verticalPadding: 3 * Theme.paddingSmall
    property string mimeType
    property bool extended: root.isSelectOneItem
    property bool singleItemSelected
    property bool virtualFolder
    property bool searchPage
    property bool homePage
    property bool pageHeaderMode
    property bool showNewFolder
    property bool isUserDir
    property bool hasFiles
    property bool isSelectionMode
    property bool isSortMode
    property bool isSelectOneItem
    property bool isArchive
    property bool isGalleryPreviewMode
    property bool standardPath
    property bool isDir
    property var appBar

    signal deleteClicked
    signal shareClicked
    signal detailsClicked
    signal renameClicked
    signal copyClicked
    signal cutClicked
    signal locationClicked
    signal compressClicked
    signal newDirectoryClicked
    signal deleteDirectoryClicked
    signal sortClicked(int sortOrder, int sortBy, int directorySort)
    signal selectModeClicked
    signal extractClicked(bool needSelection)
    signal pasteModeClicked
    signal openUrlExternallyClicked
    signal pressedOutside

    preferredHeight: Math.max(Screen.width, Screen.height)
    preferredWidth: Screen.sizeCategory > Screen.Medium
                    ?  Math.max(Screen.width, Screen.height) * 0.33333333 - Theme.horizontalPageMargin
                    :  Math.min(Screen.width, Screen.height) - Theme.horizontalPageMargin * 2

    PopupMenuItem {
        onClicked: root.deleteClicked()

        objectName: "DeleteItem"
        //% "Delete"
        text: qsTrId("filemanager-me-delete")
        visible: root.extended && !deleteFolderMenuItem.visible && !standardPath
        icon.source: "image://theme/icon-splus-delete"
    }

    PopupMenuItem {
        onClicked: root.shareClicked()

        objectName: "ShareItem"
        //% "Share"
        text: qsTrId("filemanager-me-share")
        visible: root.extended && !root.isDir
        icon.source: "image://theme/icon-splus-share"
    }

    PopupMenuItem {
        onClicked: root.copyClicked()

        objectName: "CopyItem"
        //% "Copy"
        text: qsTrId("filemanager-me-copy")
        visible: (!root.pageHeaderMode && root.extended) || root.isGalleryPreviewMode
        icon.source: "image://theme/icon-splus-copy"
    }

    PopupMenuItem {
        onClicked: root.cutClicked()

        objectName: "CutItem"
        visible: (!virtualFolder || searchPage) && !standardPath && !pageHeaderMode
        //% "Cut"
        text: qsTrId("filemanager-me-cut")
        icon.source: "image://theme/icon-splus-cut"
    }

    PopupMenuItem {
        onClicked: root.detailsClicked()

        objectName: "DetailsItem"
        //% "Details"
        text: qsTrId("filemanager-me-details")
        visible: root.singleItemSelected
        icon.source: "image://theme/icon-splus-about"
    }

    PopupMenuItem {
        onClicked: root.renameClicked()

        objectName: "RenameItem"
        //% "Rename"
        text: qsTrId("filemanager-me-rename")
        visible: root.singleItemSelected && !standardPath
        icon.source: "image://theme/icon-splus-rename"
    }

    PopupMenuItem {
        onClicked: root.compressClicked()

        objectName: "AddToArchiveItem"
        visible: !pageHeaderMode
        //% "Add to archive"
        text: qsTrId("filemanager-me-add_to_archive")
        icon.source: "image://theme/icon-splus-file-archive-folder"
    }

    PopupMenuItem {
        onClicked: root.locationClicked()

        objectName: "ObjectLocationItem"
        visible: searchPage && singleItemSelected
        //% "Object location"
        text: qsTrId("filemanager-me-object_location")
        icon.source: "image://theme/icon-splus-file-folder"
    }

    PopupSubMenuItem {
        id: submenuSort

        objectName: "SortItem"
        visible: hasFiles
                 && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("SortingButton") === -1)
                 && !homePage && (!isSelectOneItem || !isSelectionMode) && pageHeaderMode
        text: qsTrId("filemanager-he-sort")

        PopupMenuSortItems {
            onSortClicked: root.sortClicked(sortOrder, sortBy, directorySort)
        }
    }


    PopupMenuItem {
        onClicked: FileManager.gridLayoutSetting.value = !FileManager.gridLayoutSetting.value

        objectName: "ListDisplayItem"
        visible: pageHeaderMode
                 && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("GridLayoutButton") === -1)
                 && hasFiles && !isSelectOneItem && !isSelectionMode

        //% "List display"
        text: FileManager.gridLayoutSetting.value ? qsTrId("filemanager-me-list_view")
                                                    //% "Grid display"
                                                  : qsTrId("filemanager-me-grid_view")
        icon.source: FileManager.gridLayoutSetting.value
                      ? "image://theme/icon-splus-list"
                      : "image://theme/icon-splus-cage"
    }

    PopupMenuItem {
        //% "Show"
        readonly property string showHidden: qsTrId("filemanager-me-dont_show_hidden_item")
        //% "Don`t show"
        readonly property string dontShowHidden: qsTrId("filemanager-me-show_hidden_item")

        onClicked: {
            root.close()
            FileManager.hiddenItemsSetting.value = !FileManager.hiddenItemsSetting.value
        }
        objectName: "HiddenitemsItem"
        visible: isUserDir
                 && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("ShowHiddenButton") === -1)
                 && (!isSelectOneItem || !isSelectionMode) && pageHeaderMode
        //% "%1 hidden items"
        text: qsTrId("filemanager-me-hidden_items").arg(FileManager.hiddenItemsSetting.value ? dontShowHidden : showHidden)
        icon.source: FileManager.hiddenItemsSetting.value
                      ? "image://theme/icon-splus-file-hide"
                      : "image://theme/icon-splus-file-show"
    }

    PopupMenuItem {
        onClicked: root.newDirectoryClicked()

        objectName: "NewFolderItem"
        height: Theme.itemSizeSmall
        visible: !virtualFolder && pageHeaderMode && showNewFolder && isUserDir
                 && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("NewFolderButton") === -1)

        //% "New folder"
        text: qsTrId("filemanager-me-new_folder")
        icon.source: "image://theme/icon-splus-new-folder"
    }

    PopupMenuItem {
        id: deleteFolderMenuItem

        onClicked: root.deleteDirectoryClicked()

        objectName: "DeleteFolderItem"
        height: Theme.itemSizeSmall
        visible: !virtualFolder && pageHeaderMode && page.showDeleteFolder && isUserDir
                 && page.path !== page.initialPath && standardPaths.indexOf(fileModel.path) === -1

        //% "Delete folder"
        text: qsTrId("filemanager-me-delete_folder")
        icon.source: "image://theme/icon-splus-delete-folder"
    }

    PopupMenuItem {
        onClicked: root.selectModeClicked()

        objectName: "SelectMultipleItem"
        height: Theme.itemSizeSmall
        visible: (pageHeaderMode
                  && (showNewFolder || virtualFolder)
                  && hasFiles
                  && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("SelectModeButton") === -1))
                  || isSelectOneItem

        //% "Select multiple"
        text: qsTrId("filemanager-me-select_multiple")
        icon.source: "image://theme/icon-splus-multiselect"
    }

    PopupMenuItem {
        onClicked: root.extractClicked(false)

        objectName: "ExtractHereItem"
        height: Theme.itemSizeSmall
        visible: isArchive && singleItemSelected && !virtualFolder && !searchPage

        //% "Extract here"
        text: qsTrId("filemanager-me-extract_here")
        icon.source: "image://theme/icon-splus-extract-in"
    }

    PopupMenuItem {
        onClicked: root.extractClicked(true)

        objectName: "ExtractToItem"
        height: Theme.itemSizeSmall
        visible: isArchive && singleItemSelected

        //% "Extract to..."
        text: qsTrId("filemanager-me-extract_to")
        icon.source: "image://theme/icon-splus-extract-out"
    }

    PopupMenuItem {
        onClicked: root.pasteModeClicked()

        objectName: "PasteFilesItem"
        height: Theme.itemSizeSmall
        visible: pageHeaderMode && isUserDir && FileEngine.clipboardCount > 0
        text: !virtualFolder && !searchPage ? page._getPasteTextId() : ""
        icon.source: "image://theme/icon-splus-paste"
    }

    PopupMenuItem {
        readonly property string currentApp: {
            if (FileManager.isPicture(root.mimeType) || FileManager.isVideo(root.mimeType))
                //% "gallery"
                return qsTrId("filemanager-me-open_in_gallery")

            return ""
        }

        onClicked: root.openUrlExternallyClicked()

        objectName: "OpenInItem"
        visible: isGalleryPreviewMode
        //% "Open in %1"
        text: qsTrId("filemanager-me-open_in").arg(currentApp)
        icon.source: "image://theme/icon-splus-image"
    }


    PopupMenuItem {
        readonly property bool _hasHomePage: typeof _multiPanel !== "undefined"
        readonly property bool _isTablet: _hasHomePage ? _multiPanel : false
        // Service element
        visible: false

        Connections {
            target: root.inverseMouseArea

            onPressedOutside: root.pressedOutside()
        }
    }
}

