/**
 * SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import Aurora.Controls 1.0

Item {
    id: root

    property alias model: dirList.model
    property alias homeButtonIcon: storageIcon.icon
    property bool homeButtonVisible: true

    readonly property bool _hasHomePage: typeof homePage !== "undefined"
    readonly property bool _isTablet: _hasHomePage ? homePage._isTablet : false

    signal updatePosition

    objectName: "PathBarItem"
    x: Theme.paddingLarge
    width: (parent ? parent.width : Screen.width) -  2 * x
    height: Theme.iconSizeLarge

    onUpdatePosition: dirList.positionViewAtEnd()

    IconButton {
        id: storageIcon

        function pushHomePage() {
            var targetPage = pageStack.find(function(dialog) { return dialog.hasOwnProperty("__home_page") })
            if (targetPage && !root._isTablet) {
                pageStack.pop(targetPage)
            } else {
                if (__home_page != null)
                    openHomePage(SplitView.Animated, false)
            }
        }

        objectName: "StorageIcon"
        anchors {
            left: parent.left
            verticalCenter: parent.verticalCenter
        }

        width: Theme.iconSizeSmallPlus
        height: root.height
        enabled: typeof _settingsApplication === "undefined"
        icon.source: "image://theme/icon-splus-home"
        highlighted: down || !homeButtonVisible
        icon.highlightColor: homeButtonVisible ? Theme.highlightColor : Theme.secondaryHighlightColor

        onClicked: pushHomePage()
    }

    SilicaListView {
        id: dirList

        objectName: "PathBarListView"
        anchors {
            left: storageIcon.right
            right: parent.right
            verticalCenter: parent.verticalCenter
            leftMargin: Theme.paddingSmall
        }

        width: parent.width - storageIcon.width
        height: parent.height
        orientation: ListView.Horizontal
        spacing: Theme.paddingMedium
        clip: true
        delegate: Row {
            objectName: "PathBarListViewRow"
            height: parent.height

            Icon {
                objectName: "NextIndicatorIcon"
                anchors.verticalCenter: dirNameItem.verticalCenter
                width: sourceSize.width
                height: sourceSize.height
                source: "image://theme/icon-xs-next-indicator"
            }

            BackgroundItem {
                id: dirNameItem

                objectName: "DirNameItem"
                anchors.verticalCenter: parent.verticalCenter
                width: dirName.width
                height: dirName.height
                _showPress: false
                enabled: !model.isSelected
                down: fillArea.down

                // to increase the pressure area
                BackgroundItem {
                    id: fillArea

                    function pushPage() {
                        var targetPage = pageStack.find(function(page) { return page.__directory_path === model.fullPath });
                        if (targetPage) {
                            pageStack.pop(targetPage)
                        } else {
                            const currentPage = pageStack.find(function(page) { return page.__directory_path === model.path });
                            if (currentPage && !root._isTablet) {
                                pageStack.animatorReplaceAbove(currentPage, "DirectoryPage.qml", {
                                                                   initialPath: model.fullPath ,
                                                                   title: model.fullPath === StandardPaths.home ? qsTrId("files-la-device")
                                                                                                                : model.directoryName,
                                                                   path: model.fullPath,
                                                                   extractor: currentPage.extractor
                                                               });
                            } else {
                                homePage.openInternalDevicePage()
                            }
                        }
                    }

                    anchors.verticalCenter: parent.verticalCenter
                    width: dirName.width
                    height: root.height
                    enabled: dirNameItem.enabled
                    _showPress: dirNameItem._showPress

                    onClicked: pushPage()
                }

                Label {
                    id: dirName

                    objectName: "DirNameLabel"
                    leftPadding: Theme.paddingMedium
                    text: model.fullPath === StandardPaths.home ? qsTrId("files-la-device") : model.directoryName
                    font.pixelSize: Theme.fontSizeExtraSmall
                    maximumLineCount: 1
                    wrapMode: Text.NoWrap
                    color: {
                        if (model.isSelected)
                            return Theme.secondaryHighlightColor

                        return dirNameItem.highlighted ? Theme.highlightColor : Theme.primaryColor
                    }
                }
            }
        }
        onWidthChanged: positionViewAtEnd()
        onCountChanged: positionViewAtEnd()
    }

    DoubleOpacityMask { listView: dirList }
}
