/**
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import QtMultimedia 5.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Sailfish.Gallery 1.0
import Nemo.Thumbnailer 1.0
import Aurora.Controls 1.0
import Aurora.Applications.Internal 1.0 as AppInternal

Page {
    id: root

    property string imagePath
    property string videoPath

    property bool autoPlay: true

    readonly property bool isImage: root.imagePath.length !== 0
    readonly property bool isVideo: root.videoPath.length !== 0
    readonly property int _headersHorizontalAlignment: Qt.AlignLeft
    readonly property int contentHeight: __privatePageSettings.isFirstParent ? root.height - previewHeader.height - toolBar.height
                                                                             : root.height
    readonly property QtObject player: playerLoader.item ? playerLoader.item.player : null
    readonly property var currentItem: contentLoader.item
    readonly property bool hasHomePage: typeof (homePage) !== "undefined"

    property alias overlay: overlay
    property bool pasteMode
    property bool extractMode
    property ExtractorView extractor

    signal endedPlayback(bool playing)

    clip: status !== PageStatus.Active || pageStack.dragInProgress
    palette.colorScheme: __privatePageSettings.isFirstParent ? Theme.colorScheme
                                                             : Theme.LightOnDark

    Component.onCompleted: __privatePageSettings.firstParent = root.parent

    FileInfo {
        id: info

        file: root.isImage ? root.imagePath
                           : root.videoPath
    }

    Rectangle {
        anchors.fill: parent
        z: -1
        color: "black"
        visible: !__privatePageSettings.isFirstParent
    }

    QtObject {
        id: __privatePageSettings

        property var firstParent
        readonly property  bool isFirstParent: root.parent === __privatePageSettings.firstParent
    }

    AppBar {
        id: previewHeader

        opacity: __privatePageSettings.isFirstParent ? 1 : 0
        objectName: "PreviewHeader"
        //% "Image preview"
        headerText: root.isImage ? qsTrId("filemanager-he-image-preview")
                                   //% "Video preview"
                                 : qsTrId("filemanager-he-video-preview")
    }

    Loader {
        id: contentLoader

        anchors.centerIn: parent
        width: parent.width
        height: root.contentHeight

        active: root.isImage || root.isVideo
        sourceComponent: root.isImage ? imageComponent : videoComponent
    }

    Component {
        id: imageComponent

        ImageViewer {
            source: root.imagePath

            Binding {
                when: photo.status === Image.Ready
                target: largePhoto
                property: "source"
                value: photo.source
            }

            onZoomedChanged: overlay.active = !zoomed
            onClicked: {
                if (zoomed) {
                    zoomOut()
                } else {
                    overlay.active = !overlay.active && !__privatePageSettings.isFirstParent
                }
            }
        }
    }

    Component {
        id: videoComponent

        VideoPoster {
            id: videoPoster

            contentWidth: parent.width
            contentHeight: parent.height

            source: root.videoPath
            mimeType: info.mimeType
            playing: root.player && root.player.playing
            loaded: root.player && root.player.loaded
            overlayMode: root.overlay.active
            busy: root.player && !root.player.hasVideo && !root.player.hasError

            onBusyChanged: {
                if (!busy)
                    busy = false
            }
            onClicked: root.overlay.active = !root.overlay.active
            onDoubleClicked: root.overlay.seekForward()
            onTogglePlay: {
                playerLoader.active = true
                root.player.togglePlay()
                playerLoader.z = 1
            }

            Connections {
                target: root
                onEndedPlayback: {
                    videoPoster.busy = false
                    videoPoster.playing = playing

                    if (root.player.status === MediaPlayer.EndOfMedia) {
                        videoPoster.loaded = false
                        playerLoader.z = -1
                    }
                }
            }

            Loader {
                anchors.centerIn: parent
                z: -1
                active: videoPoster.status === Thumbnail.Error
                width: parent.width
                height: Math.min(Theme.itemSizeHuge * 4, root.contentHeight / 1.5)
                sourceComponent: videoWithoutPreviewComponent
            }
        }
    }

    Loader {
        id: playerLoader

        anchors.centerIn: parent
        z: -1
        width: parent.width
        height: root.contentHeight
        active: false

        sourceComponent: GalleryVideoOutput {
            width: parent.width
            height: parent.height
            player: GalleryMediaPlayer {
                id: mediaPlayer

                active: !root.isImage && Qt.application.active
                source: active ? root.videoPath : ""
                onPlayingChanged: {
                    root.endedPlayback(playing)

                    if (playing && root.overlay.active) {
                        // go fullscreen for playback if triggered via Play icon.
                        root.overlay.active = false
                    }
                }

                onDisplayError: root.currentItem.displayError()
            }

            MouseArea {
                anchors.fill: parent
                onClicked: {
                    root.overlay.active = !root.overlay.active
                }
            }

            Image {
                id: icon
                anchors.centerIn: parent
                enabled: root.overlay.active
                visible: enabled

                Binding	{
                    target: icon
                    when: overlay.active
                    property: "source"
                    value: "image://theme/icon-video-overlay-" + (root.player.playing ?  "pause" : "play")
                           + "?" + (iconMouse.down ? Theme.highlightColor : Theme.lightPrimaryColor)
                }
                MouseArea {
                    id: iconMouse

                    property bool down: pressed && containsMouse
                    anchors.fill: parent
                    onClicked: root.player.togglePlay()
                }
            }
        }
    }

    Component {
        id: videoWithoutPreviewComponent

        Rectangle {
            anchors.fill: parent
            color: "black"
        }
    }

    GalleryOverlay {
        id: overlay

        anchors {
            fill: parent
            bottomMargin: __privatePageSettings.isFirstParent ? toolBar.height
                                                              : 0
            topMargin: __privatePageSettings.isFirstParent ? previewHeader.height
                                                           : 0
        }

        z: previewHeader.z + 1
        active: !__privatePageSettings.isFirstParent
        player: root.player
        source: info.absolutePath
        isImage: root.isImage
        error: root.currentItem && root.currentItem.error
        detailsButton.visible: false
        editingAllowed: false
        deletingAllowed: false
        sharingAllowed: false
        ambienceAllowed: false
        printingAllowed: false

        onCreatePlayer: playerLoader.active = true

        Private.DismissButton {
            anchors {
                top: parent.top
                topMargin: Theme.paddingLarge + SafeZoneRect.insets.top
                right: parent.right
                rightMargin: Theme.horizontalPageMargin + Math.max(SafeZoneRect.insets.left, SafeZoneRect.insets.right)
            }

            visible: overlay.active && !__privatePageSettings.isFirstParent
            popPageOnClick: false

            onClicked: {
                 __silica_applicationwindow_instance.displayMode = toolBar.__defaultScreenMode
                root.parent = __privatePageSettings.firstParent
                overlay.active = false
                root.backNavigation = true
            }
        }
    }

    FileGridFooter {
        id: toolBar

        property int __defaultScreenMode

        _bottomPosition: true
        visible: __privatePageSettings.isFirstParent
        parentGridView: root
        isGalleryPreviewMode: true

        onShareClicked: overlay.triggerAction("share", true)
        onDeleteClicked: {
            popupMenu.playerPause()
            root.extractor.openThirdColumn(Qt.resolvedUrl("RemoveEntityDialog.qml"), {
                                               filePaths: [info.absolutePath],
                                               acceptedCallback: function(dialog) {
                                                   root.opacity = Theme.opacityFaint
                                                   removeIndicator.running = true
                                                   FileEngine.deleteFiles(dialog.filePaths, true)
                                                   dialog.displayResultNotice()
                                               },
                                               destructionCallback: function(dialog) {
                                                   if (!dialog.wasAccepted) {
                                                       return
                                                   }

                                                   AppInternal.Utils.setTimeout(function() {
                                                       removeIndicator.running = false

                                                   }, 400)
                                                   pageStack.pop()
                                               }
                                           })
        }
        onFullScreenModeClicked: {
            __silica_applicationwindow_instance.displayMode = ApplicationDisplayMode.FillScreen

            if (hasHomePage) {
                root.parent = homePage
            } else {
                root.parent = pageStack
                root.backNavigation = false
            }
        }
        onOpenMenuClicked: {
            popupMenu.open(toolBar.itemAnchorPopup)
        }
        onCancelClicked: {
            FileEngine.clearClipboard()
            root.pasteMode = false
        }

        Component.onCompleted: toolBar.__defaultScreenMode = __silica_applicationwindow_instance.displayMode
    }

    PageBusyIndicator {
        id: removeIndicator
    }

    TouchBlocker {
        anchors.fill: parent
        enabled: removeIndicator.running
    }

    PopupMenuMain {
        id: popupMenu

        function playerPause() {
            if (root.player && root.isVideo) {
                root.player.pause()
            }
        }


        onOpenUrlExternallyClicked: Qt.openUrlExternally(FileManager.pathToUrl(info.absolutePath))
        onCutClicked: {
            root.pasteMode = true
            FileEngine.cutFiles([info.absolutePath])
        }
        onDetailsClicked: {
            playerPause()
            root.extractor.openThirdColumn(Qt.resolvedUrl("DetailsPage.qml"), {
                                               fileName: info.fileName,
                                               mimeType: info.mimeType,
                                               isDir: info.isDir,
                                               modified: info.modified,
                                               size: info.size,
                                               path: info.absolutePath
                                           })
        }

        onRenameClicked: {
            playerPause()
            root.extractor.openThirdColumn(Qt.resolvedUrl("RenameDialog.qml"), {
                                               oldPath: info.absolutePath,
                                               acceptText: qsTrId("filemanager-he-rename")
                                           })
        }

        onCopyClicked: {
            root.pasteMode = true
            FileEngine.copyFiles([info.absolutePath])
        }

        onCompressClicked: {
            playerPause()
            root.extractor.openThirdColumn(Qt.resolvedUrl("CreateArchiveDialog.qml"), {
                                               extractorView: root.extractor,
                                               files: [info.absolutePath],
                                               currentDirectoryPath: info.directoryPath
                                           })
        }

        z: overlay.z + 1
        hasFiles: root.isImage || root.isVideo
        isUserDir: FileEngine.isUserDir(info.directoryPath)
        isGalleryPreviewMode: true
        singleItemSelected: true
        mimeType: info.mimeType
        appBar: appBar
    }

    Connections {
        target: FileEngine
        onFileRenamed: info.file = newPath
    }
}
