/*
 * SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2018 Jolla Ltd.
 *
 * License: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0

BusyView {
    id: root

    property alias model: archiveModel
    property alias path: archiveModel.path
    property string archiveFile
    property alias fileName: archiveModel.fileName
    property alias baseExtractionDirectory: archiveModel.archiveDirPath
    property alias isOpen: archiveModel.isOpen
    property bool passwordNeeded
    property bool hasIncorrectPassword
    property string destinationPageName
    property string rejectedPageName
    property bool extractHereAvailable: true
    property bool isInternalMode: true
    property bool extractMode

    readonly property bool hasHomePage: typeof homePage !== "undefined"
    readonly property PageStack _currentPageStack: {
        if (root.hasHomePage)
            return !homePage._isTablet ? pageStack : splitView.pageStack

        return pageStack
    }

    signal archiveExtracted(string containingFolder)
    signal showInfo(string info)

    function extractAllFiles(targetPath) {
        var target = _buildExtractionDirectory(false, true, model.baseName, targetPath)
        return archiveModel.extractAllFiles(target)
    }

    function extractFile(targetPath, fileName, isDir) {
        var targetDir = _buildExtractionDirectory(isDir, false, fileName, targetPath)
        return archiveModel.extractFile(fileName, targetDir)
    }

    function appendPath(fileName) {
        return archiveModel.appendPath(fileName)
    }

    function compressFiles(files, archiveName, targetPath, archiveType, password) {
        return archiveModel.compressFiles(files, archiveName, targetPath, archiveType, password)
    }

    function _buildExtractionDirectory(isDir, isArchive, dirName, targetPath) {
        var extractionDirectory = targetPath ? targetPath : baseExtractionDirectory
        return extractionDirectory
    }

    function openThirdColumn(url, properties) {
        if (root.hasHomePage) {
            return openDialogPage(url, properties)
        } else {
            return pageStack.animatorPush(url, properties)
        }
    }

    function reset(args) {
        args = args || {}

        root.isOpen = false
        root.passwordNeeded = false

        if (!root.extractMode || args.force) {
            root.archiveFile = ""
            root.path = ""
        }
    }

    function selectedExtractPath(extract, fileName, isDir) {
        const obj = openThirdColumn(Qt.resolvedUrl("FolderPickerPage.qml"))
        obj.pageCompleted.connect(function(picker) {
            picker.selectedPathChanged.connect(function() {
                if (picker.selectedPath.length > 0) {
                    const rejectedPage = root._currentPageStack.find(function(dialog) { return dialog.hasOwnProperty(root.rejectedPageName) })
                    if (root.hasHomePage) {
                        dialogPageStack.clear()
                    }
                    root._currentPageStack.pop(rejectedPage)
                    extract(picker.selectedPath, fileName, isDir)
                } else {
                    //% "Extraction path is incorrect"
                    showInfo(qsTrId("filemanager-la-extraction_path_incorrect"))
                }
            })
        })
    }

    onArchiveFileChanged: archiveModel.archiveFile = archiveFile

    // Grace timer
    Timer {
        id: graceTimer

        interval: 500
        running: model.extracting || model.compressing
        onTriggered: {
            if (passwordNeeded) {
                 var properties = {
                    extractorView: root,
                    passwordError: root.hasIncorrectPassword,
                    destinationPageName: root.destinationPageName,
                    acceptDestination: root.archivePageComponent
                }

                const pageName = Qt.resolvedUrl("ArchivePasswordDialog.qml")
                const currentPageStack = root.hasHomePage
                                       ? !homePage._isTablet
                                         ? pageStack
                                         : splitView.pageStack
                                       : pageStack
                const rejectedPage = currentPageStack.find(function(dialog) { return dialog.hasOwnProperty(root.rejectedPageName) })

                const obj = openThirdColumn(pageName, properties)
                obj.pageCompleted.connect(function(dialog) {
                    dialog.onRejected.connect(function() {
                        archiveModel.cleanAction()
                        if (root.hasHomePage) {
                            dialogPageStack.clear()
                        }
                        currentPageStack.pop(rejectedPage)
                    })
                })

                graceTimer.stop()
                root.hasIncorrectPassword = false
            }
        }
    }

    busy: (model.extracting || model.compressing) && !graceTimer.running && !passwordNeeded
    enabled: busy
    text: {
        if (model.extracting) {
            //% "Extracting archive %1"
            return qsTrId("filemanager-la-extracting_archive").arg(archiveModel.operationProgress + "%")
        }

        if (model.compressing) {
            //% "Compressing archive %1"
            return qsTrId("filemanager-la-compressing_archive").arg(archiveModel.operationProgress + "%")
        }

        return ""
    }

    ArchiveModel {
        id: archiveModel

        property int operationProgress: 0

        autoRename: true
        onErrorStateChanged: {
            switch (errorState) {
            case ArchiveModel.ErrorUnsupportedArchiveFormat:
                //% "Unsupported archive format"
                showInfo(qsTrId("filemanager-la-unsupported_archive_format"))
                break
            case ArchiveModel.ErrorArchiveNotFound:
                //% "Archive file is not found"
                showInfo(qsTrId("filemanager-la-archive_not_found"))
                break
            case ArchiveModel.ErrorArchiveOpenFailed:
                //% "Opening archive failed"
                showInfo(qsTrId("filemanager-la-opening_archive_failed"))
                FileManager.openArchiveIsCorrupted = true
                break
            case ArchiveModel.ErrorArchiveExtractFailed:
                //% "Extract failed"
                showInfo(qsTrId("filemanager-la-extract_failed"))
                break
            case ArchiveModel.ErrorExtractingInProgress:
                //% "Extracting in progress"
                showInfo(qsTrId("filemanager-la-extracting_in_progress"))
                break
            case ArchiveModel.ErrorCompressingInProgress:
                //% "Compressing in progress"
                showInfo(qsTrId("filemanager-la-сompressing_in_progress"))
                break
            case ArchiveModel.ErrorFilesToCompressNotFound:
                //% "No files to compress"
                showInfo(qsTrId("filemanager-la-сompress_no_files"))
                break
            case ArchiveModel.ErrorInvalidArchiveName:
                //% "Invalid archive name"
                showInfo(qsTrId("filemanager-la-сompress_invalid_archive_name"))
                break
            case ArchiveModel.ErrorMissingDirForCreateArchive:
                //% "Missing directory for create archive"
                showInfo(qsTrId("filemanager-la-сompress_missing_directory"))
                break
            case ArchiveModel.ErrorMissingPathForCreateArchive:
                //% "Missing path for create archive"
                showInfo(qsTrId("filemanager-la-сompress_missing_path"))
                break
            case ArchiveModel.ErrorInvalidArchiveType:
                //% "Invalid archive type"
                showInfo(qsTrId("filemanager-la-сompress_invalid_archive_type"))
                break
            case ArchiveModel.ErrorFileCompresstFailed:
                //% "Compress files failed"
                showInfo(qsTrId("filemanager-la-compress_files_failed"))
                break
            case ArchiveModel.UnknownError:
                //% "Operation failed"
                showInfo(qsTrId("filemanager-la-creating_archive_failed"))
                break
            }
        }

        onFilesExtracted: {
            if (isDir) {
                //% "Directory %1 extracted"
                showInfo(qsTrId("filemanager-la-directory_extracted").arg(entryName))

                var title = path.substring(StandardPaths.home.length + 1)
                var directory = root._currentPageStack.replace("Sailfish.FileManager.DirectoryPage", {
                                                      path: path,
                                                      title: title,
                                                      initialPath: path,
                                                      showDeleteFolder: true,
                                                      //% "Extracted folder"
                                                      description: qsTrId("sailfish-archive-he-extract_folder"),
                                                      extractor: root
                                                  }, PageStackAction.Immediate)
            } else if (entries && entries.length === 1) {
                //% "Extracted %1"
                showInfo(qsTrId("filemanager-la-file_extracted").arg(entryName))
            } else {
                //% "%1 extracted"
                showInfo(qsTrId("filemanager-la-archive_extracted").arg(fileName))
                root.archiveExtracted(path)
            }
            archiveModel.operationProgress = 0
        }

        onFilesCompressed: {
            //% "%1 created"
            showInfo(qsTrId("filemanager-la-file_compress_created").arg(archiveName))
            archiveModel.operationProgress = 0
        }

        onProgressChanged: archiveModel.operationProgress = percent

        onPasswordRequested: {
            root.passwordNeeded = true
            graceTimer.start()
        }

        onIncorrectPassword: {
            root.passwordNeeded = true
            root.hasIncorrectPassword = true
            graceTimer.start()
        }
    }

    Connections {
        target: Qt.application
        onAboutToQuit: {
            archiveModel.cleanAction(archiveModel.extracting || archiveModel.compressing)
        }
    }
}
