/*
 * SPDX-FileCopyrightText: 2016-2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2019-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Nemo.Thumbnailer 1.0

Page {
    id: root

    property string fileName
    property string mimeType
    property bool isDir
    property date modified
    property double size
    property string path
    property int itemCount

    readonly property int _headersHorizontalAlignment: Qt.AlignLeft

    ListModel {
        id: detailsModel

        readonly property string contentsLabel: {
            du.status !== DiskUsage.Idle
            //: Shown when counting number of items a directory (context menu -> details)
            //% "Counting…"
                    ? qsTrId("filemanager-la-counting")
                      //% "%n items, totalling %1."
                    : qsTrId("filemanager-la-items", root.itemCount).arg(Format.formatFileSize(root.size))
        }

        function updateContentsLabel() {
            for (var i = 0; i < detailsModel.count; ++i) {
                if (detailsModel.get(i).isContentsLabel) {
                    detailsModel.setProperty(i, "value", detailsModel.contentsLabel)
                    return
                }
            }
        }

        function addDetail(label, value, isVisible, isContentsLabel) {
            if (isContentsLabel === undefined)
                isContentsLabel = false

            detailsModel.append({
                                    "label": label,
                                    "value": value,
                                    "isVisible": isVisible,
                                    "isContentsLabel": isContentsLabel
                                })
        }

        dynamicRoles: true
    }

    SilicaFlickable {
        objectName: "DetailsPageSilicaFlickable"

        anchors.fill: parent
        contentHeight: column.height + Theme.paddingMedium

        Column {
            id: column

            width: parent.width
            spacing: Theme.paddingLarge

            PageHeader {
                objectName: "InformationPageHeader"

                //% "Directory information"
                title: isDir ? qsTrId("filemanager-he-dir-info")
                               //% "File information"
                             : qsTrId("filemanager-he-file-info")
            }

            DetailsIcon {
                objectName: "DetailIcon"

                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width - 2 * Theme.horizontalPageMargin
                backgroundWidth: Theme.iconSizeExtraLarge
                backgroundRadius: Theme.paddingMedium
                isDir: root.isDir
                mimeType: root.mimeType
                path: root.path
                needToDisplayIcon: false
            }

            Repeater {
                model: detailsModel
                delegate: DetailItem {
                    objectName: "FileDetailItem"

                    label: model.label
                    value: model.value
                    visible: model.isVisible
                    forceValueBelow: true
                    alignment: Qt.AlignLeft
                    _valueItem.font.pixelSize: Theme.fontSizeMedium
                    _valueItem.color: Theme.primaryColor
                }
            }
        }

        VerticalScrollDecorator {}
    }

    DiskUsage { id: du }

    Component.onCompleted: {
        if (isDir) {
            du.calculate(path, function (usage) {
                size = usage[path]
                detailsModel.updateContentsLabel()
            })

            du.fileCount(path, function(count) {
                itemCount = count
                detailsModel.updateContentsLabel()
            }, DiskUsage.Files | DiskUsage.Dirs)
        }

        //% "Name"
        detailsModel.addDetail(qsTrId("filemanager-he-name"),
                               root.fileName,
                               true)
        //% "Path to %1"
        detailsModel.addDetail(qsTrId("filemanager-he-path").arg(root.isDir
                                                                   //% "folder"
                                                                 ? qsTrId("filemanager-he-path_folder")
                                                                   //% "file"
                                                                 : qsTrId("filemanager-he-path_file")),
                               root.path,
                               true)
        //% "Size"
        detailsModel.addDetail(qsTrId("filemanager-he-size"),
                               Format.formatFileSize(root.size),
                               !root.isDir)
        //% "Contents"
        detailsModel.addDetail(qsTrId("filemanager-he-contents"),
                               detailsModel.contentsLabel,
                               root.isDir,
                               true)
        //% "Type"
        detailsModel.addDetail(qsTrId("filemanager-he-type"),
                               root.mimeType,
                               true)
        //% "Modified"
        detailsModel.addDetail(qsTrId("filemanager-he-modified"),
                               Format.formatDate(root.modified, Formatter.DateLong),
                               true)
    }
}
