/*
 * SPDX-FileCopyrightText: 2020-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef USERMODEL_H
#define USERMODEL_H

#include <QAbstractListModel>
#include <QDBusError>
#include <QHash>
#include <QSet>
#include <QVector>

#include "systemsettingsglobal.h"
#include "userinfo.h"

#include <nemo-dbus/interface.h>
#include <sailfishusermanagerinterface.h>

class QDBusServiceWatcher;
struct SailfishUserManagerEntry;

class SYSTEMSETTINGS_EXPORT UserModel: public QAbstractListModel
{
    Q_OBJECT
    Q_PROPERTY(bool placeholder READ placeholder WRITE setPlaceholder NOTIFY placeholderChanged)
    Q_PROPERTY(int count READ count NOTIFY countChanged)
    Q_PROPERTY(int maximumCount READ maximumCount NOTIFY maximumCountChanged)

public:
    enum Roles {
        UsernameRole = Qt::UserRole,
        NameRole,
        TypeRole,
        UidRole,
        CurrentRole,
        PlaceholderRole,
        TransitioningRole,
    };
    Q_ENUM(Roles)

    enum UserType {
        User = 0,
        DeviceOwner = 1,
    };
    Q_ENUM(UserType)

    enum ErrorType {
        Failure = QDBusError::Failed,
        OtherError = QDBusError::Other,
        InvalidArgs = QDBusError::InvalidArgs,
        Busy = 100,
        HomeCreateFailed,
        HomeRemoveFailed,
        GroupCreateFailed,
        UserAddFailed,
        UserModifyFailed,
        UserRemoveFailed,
        GetUidFailed,
        UserNotFound,
        AddToGroupFailed,
        RemoveFromGroupFailed,
        MaximumNumberOfUsersReached,
        AnotherUserIsAdding
    };
    Q_ENUM(ErrorType)

    enum QuotaStatus {
        QuotaStatusNoError = QuotaErrorType::NoError,
        QuotaStatusTurnedOff = QuotaErrorType::TurnedOff,
        QuotaStatusSystemError = QuotaErrorType::SystemError,
        QuotaStatusHasUsers = QuotaErrorType::HasUsers,
        QuotaStatusSmallVolume = QuotaErrorType::SmallVolume,
        QuotaStatusWrongValue = QuotaErrorType::WrongValue,
        QuotaStatusDBusError = 65536
    };
    Q_ENUM(QuotaStatus)

    explicit UserModel(QObject *parent = 0);
    ~UserModel();

    bool placeholder() const;
    void setPlaceholder(bool value);
    int count() const;
    int maximumCount() const;

    QHash<int, QByteArray> roleNames() const;
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;
    bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole);
    QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const;

    // Methods to modify users
    Q_INVOKABLE void createUser();
    Q_INVOKABLE void createUser(const QVariantMap &param);
    Q_INVOKABLE void removeUser(int row);
    Q_INVOKABLE void reset(int row);
    Q_INVOKABLE void setCurrentUser(int row);
    Q_INVOKABLE UserInfo * getCurrentUser() const;

    // Methods to modify user's groups
    Q_INVOKABLE bool hasGroup(int row, const QString &group) const;
    Q_INVOKABLE void addGroups(int row, const QStringList &groups);
    Q_INVOKABLE void removeGroups(int row, const QStringList &groups);
    Q_INVOKABLE void getUserQuotaLimits();
    Q_INVOKABLE void getUserQuotaInfo();
    Q_INVOKABLE void checkQuotaOperationsPossible(bool usersExist);
    Q_INVOKABLE void enableUsersQuota(bool enable);
    Q_INVOKABLE bool userNameCheck(const QString &name);
    Q_INVOKABLE bool userLoginCheck(const QString &login);

signals:
    void placeholderChanged();
    void countChanged();
    void maximumCountChanged();
    void userGroupsChanged(int row);
    void userAddFailed(int error);
    void userModifyFailed(int row, int error);
    void userRemoveFailed(int row, int error);
    void setCurrentUserFailed(int row, int error);
    void addGroupsFailed(int row, int error);
    void removeGroupsFailed(int row, int error);
    void userQuotaLimitsReceived(double min, double max);
    void userQuotaInfoReceived(QVariantMap info);
    void quotaOperationsPossibleReceived(uint result);
    void userCreated(int uid);

private slots:
    void onUserAdded(const SailfishUserManagerEntry &entry);
    void onUserModified(uint uid, const QString &newName);
    void onUserRemoved(uint uid);
    void onCurrentUserChanged(uint uid);
    void onCurrentUserChangeFailed(uint uid);
    void onUserQuotaInfoChanged(QVariantMap info);
    void onUserCreateFailed(const QString &error, const QString &message);
    void onUserRemoveFailed(const QString &error, const QString &message, uint uid);

    void createInterface();
    void destroyInterface();

private:
    void add(UserInfo &user);
    void onUserRemoveFailed(int errorType, const QString &message, uint uid);

    QVector<UserInfo> m_users;
    QHash<uint, int> m_uidsToRows;
    QSet<uint> m_transitioning;
    NemoDBus::Interface *m_dBusInterface;
    QDBusServiceWatcher *m_dBusWatcher;
    QSet<QString> m_userLogins;
};
#endif /* USERMODEL_H */
