/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef SAILFISH_MDM_RUNTIMEMANAGER_H
#define SAILFISH_MDM_RUNTIMEMANAGER_H

#include <QObject>
#include <QScopedPointer>
#include <QVariant>
#include <QVector>

#include <tuple>
#include <RuntimeManager/Control>

namespace Sailfish {
namespace Mdm {

class RuntimeManagerPrivate;
class Q_DECL_EXPORT RuntimeManager : public QObject
{
    Q_OBJECT

public:
    enum class StartReason {
        Unknown = 0,
        Manual,
        Autostart,
        Intent,
    };

    enum ApplicationState {
        Starting = 0,
        Active,
        Inactive,
        Hidden,
        Terminating,
        Invalid = 255,
    };

    struct RunningApplication
    {
        QString appInstanceId;
        quint32 pid;
        StartReason startReason;
        ApplicationState state;
        QVariantMap details;
    };

    struct RunningService
    {
        QString appId;
        QString name;
        quint32 pid;
        StartReason startReason;
        QVariantMap details;
    };

    explicit RuntimeManager(QObject *parent = Q_NULLPTR);
    ~RuntimeManager() override;

    std::tuple<QString, QVariantMap> start(const QString &appId);
    bool terminate(const QString &appInstanceId);
    bool getRunningApplications(QVector<RunningApplication> &apps,
                                QVector<RunningService> &services);
    QString getAppIdByPid(const quint32 &pid);

    QString lastErrorName() const;
    QString lastErrorMessage() const;

Q_SIGNALS:
    void started(const QString &appInstanceId, const QVariantMap &details);
    void terminated(const QString &appInstanceId, const QVariantMap &details);

private:
    Q_DISABLE_COPY(RuntimeManager)
    Q_DECLARE_PRIVATE(RuntimeManager)

    QScopedPointer<RuntimeManagerPrivate> d_ptr;
};

} // namespace Mdm
} // namespace Sailfish

#endif // SAILFISH_MDM_RUNTIMEMANAGER_H
